#ifndef MESH_DATA_BASE_FOR_TESTS_HPP
#define MESH_DATA_BASE_FOR_TESTS_HPP

#include <mesh/IMesh.hpp>

template <size_t Dimension>
class Connectivity;

template <typename ConnectivityT>
class Mesh;

#include <array>
#include <memory>
#include <string>

class MeshDataBaseForTests
{
 public:
  template <size_t Dimension>
  class NamedMesh
  {
   private:
    const std::string m_name;
    const std::shared_ptr<const Mesh<Connectivity<Dimension>>> m_mesh;

   public:
    NamedMesh(const std::string& name, const std::shared_ptr<const Mesh<Connectivity<Dimension>>>& mesh)
      : m_name(name), m_mesh(mesh)
    {}

    const std::string&
    name() const
    {
      return m_name;
    }

    auto
    mesh() const
    {
      return m_mesh;
    }
  };

 private:
  explicit MeshDataBaseForTests();

  static const MeshDataBaseForTests* m_instance;

  std::shared_ptr<const Mesh<Connectivity<1>>> m_cartesian_1d_mesh;
  std::shared_ptr<const Mesh<Connectivity<2>>> m_cartesian_2d_mesh;
  std::shared_ptr<const Mesh<Connectivity<3>>> m_cartesian_3d_mesh;

  std::shared_ptr<const Mesh<Connectivity<1>>> m_unordered_1d_mesh;
  std::shared_ptr<const Mesh<Connectivity<2>>> m_hybrid_2d_mesh;
  std::shared_ptr<const Mesh<Connectivity<3>>> m_hybrid_3d_mesh;

  std::shared_ptr<const Mesh<Connectivity<1>>> _buildUnordered1dMesh();
  std::shared_ptr<const Mesh<Connectivity<2>>> _buildHybrid2dMesh();
  std::shared_ptr<const Mesh<Connectivity<3>>> _buildHybrid3dMesh();

 public:
  std::shared_ptr<const Mesh<Connectivity<1>>> cartesian1DMesh() const;
  std::shared_ptr<const Mesh<Connectivity<1>>> unordered1DMesh() const;

  std::shared_ptr<const Mesh<Connectivity<2>>> cartesian2DMesh() const;
  std::shared_ptr<const Mesh<Connectivity<2>>> hybrid2DMesh() const;

  std::shared_ptr<const Mesh<Connectivity<3>>> cartesian3DMesh() const;
  std::shared_ptr<const Mesh<Connectivity<3>>> hybrid3DMesh() const;

  static const MeshDataBaseForTests& get();

  auto
  all1DMeshes() const
  {
    return std::array{NamedMesh{"cartesian 1d mesh", cartesian1DMesh()},   //
                      NamedMesh{"unordered 1d mesh", unordered1DMesh()}};
  }

  auto
  all2DMeshes() const
  {
    return std::array{NamedMesh{"cartesian 2d mesh", cartesian2DMesh()},   //
                      NamedMesh{"hybrid 2d mesh", hybrid2DMesh()}};
  }

  auto
  all3DMeshes() const
  {
    return std::array{NamedMesh{"cartesian 3d mesh", cartesian3DMesh()},   //
                      NamedMesh{std::string("hybrid 3d mesh"), hybrid3DMesh()}};
  }

  static void create();
  static void destroy();

  ~MeshDataBaseForTests() = default;
};

#endif   // MESH_DATA_BASE_FOR_TESTS_HPP
