#include <MeshDataBaseForTests.hpp>
#include <mesh/CartesianMeshBuilder.hpp>
#include <mesh/Connectivity.hpp>
#include <mesh/GmshReader.hpp>
#include <utils/Messenger.hpp>
#include <utils/PugsAssert.hpp>

#include <filesystem>
#include <fstream>

const MeshDataBaseForTests* MeshDataBaseForTests::m_instance = nullptr;

MeshDataBaseForTests::MeshDataBaseForTests()
{
  m_cartesian_1d_mesh = std::dynamic_pointer_cast<const Mesh<Connectivity<1>>>(
    CartesianMeshBuilder{TinyVector<1>{-1}, TinyVector<1>{3}, TinyVector<1, size_t>{23}}.mesh());

  m_cartesian_2d_mesh = std::dynamic_pointer_cast<const Mesh<Connectivity<2>>>(
    CartesianMeshBuilder{TinyVector<2>{0, -1}, TinyVector<2>{3, 2}, TinyVector<2, size_t>{6, 7}}.mesh());

  m_cartesian_3d_mesh = std::dynamic_pointer_cast<const Mesh<Connectivity<3>>>(
    CartesianMeshBuilder{TinyVector<3>{0, 1, 0}, TinyVector<3>{2, -1, 3}, TinyVector<3, size_t>{6, 7, 4}}.mesh());

  m_unordered_1d_mesh = _buildUnordered1dMesh();
  m_hybrid_2d_mesh    = _buildHybrid2dMesh();
  m_hybrid_3d_mesh    = _buildHybrid3dMesh();
}

const MeshDataBaseForTests&
MeshDataBaseForTests::get()
{
  return *m_instance;
}

void
MeshDataBaseForTests::create()
{
  Assert(m_instance == nullptr);
  m_instance = new MeshDataBaseForTests();
}

void
MeshDataBaseForTests::destroy()
{
  Assert(m_instance != nullptr);
  delete m_instance;
  m_instance = nullptr;
}

std::shared_ptr<const Mesh<Connectivity<1>>>
MeshDataBaseForTests::cartesian1DMesh() const
{
  return m_cartesian_1d_mesh;
}

std::shared_ptr<const Mesh<Connectivity<2>>>
MeshDataBaseForTests::cartesian2DMesh() const
{
  return m_cartesian_2d_mesh;
}

std::shared_ptr<const Mesh<Connectivity<3>>>
MeshDataBaseForTests::cartesian3DMesh() const
{
  return m_cartesian_3d_mesh;
}

std::shared_ptr<const Mesh<Connectivity<1>>>
MeshDataBaseForTests::unordered1DMesh() const
{
  return m_unordered_1d_mesh;
}

std::shared_ptr<const Mesh<Connectivity<2>>>
MeshDataBaseForTests::hybrid2DMesh() const
{
  return m_hybrid_2d_mesh;
}

std::shared_ptr<const Mesh<Connectivity<3>>>
MeshDataBaseForTests::hybrid3DMesh() const
{
  return m_hybrid_3d_mesh;
}

std::shared_ptr<const Mesh<Connectivity<1>>>
MeshDataBaseForTests::_buildUnordered1dMesh()
{
  const std::string filename = std::filesystem::path{PUGS_BINARY_DIR}.append("tests").append("unordered-1d.msh");
  if (parallel::rank() == 0) {
    std::ofstream fout(filename);
    fout << R"($MeshFormat
2.2 0 8
$EndMeshFormat
$PhysicalNames
5
0 1 "XMIN"
0 2 "XMAX"
0 3 "INTERFACE"
1 4 "LEFT"
1 5 "RIGHT"
$EndPhysicalNames
$Nodes
35
1 0 0 0
2 -1 0 0
3 1 0 0
4 0.03246387076421061 0 0
5 0.07692961499792127 0 0
6 0.1378343163027127 0 0
7 0.2212554508344625 0 0
8 0.3355173320316702 0 0
9 0.4920217492706127 0 0
10 0.7063857794730655 0 0
11 -0.9513615311521172 0 0
12 -0.9036693300347614 0 0
13 -0.8569049934960202 0 0
14 -0.8110504814343311 0 0
15 -0.7660880891035384 0 0
16 -0.7220004662902741 0 0
17 -0.6787705699229041 0 0
18 -0.6363817403249178 0 0
19 -0.5948175927947426 0 0
20 -0.5540620983944274 0 0
21 -0.5140995098187926 0 0
22 -0.4749144219806904 0 0
23 -0.4364916774353312 0 0
24 -0.3988164798351664 0 0
25 -0.3618742567441507 0 0
26 -0.3256507606537927 0 0
27 -0.2901320091863738 0 0
28 -0.2553042883413312 0 0
29 -0.2211541538737678 0 0
30 -0.1876684263237806 0 0
31 -0.1548341743437643 0 0
32 -0.1226387276497952 0 0
33 -0.09106965120674937 0 0
34 -0.0601147701140996 0 0
35 -0.02976212372405818 0 0
$EndNodes
$Elements
37
1 15 2 3 1 1
2 15 2 1 2 2
3 15 2 2 3 3
4 1 2 5 1 1 4
5 1 2 5 1 4 5
6 1 2 5 1 5 6
7 1 2 5 1 6 7
8 1 2 5 1 7 8
9 1 2 5 1 8 9
10 1 2 5 1 9 10
11 1 2 5 1 10 3
12 1 2 4 2 2 11
13 1 2 4 2 11 12
14 1 2 4 2 12 13
15 1 2 4 2 13 14
16 1 2 4 2 14 15
17 1 2 4 2 15 16
18 1 2 4 2 16 17
19 1 2 4 2 17 18
20 1 2 4 2 18 19
21 1 2 4 2 19 20
22 1 2 4 2 20 21
23 1 2 4 2 21 22
24 1 2 4 2 22 23
25 1 2 4 2 23 24
26 1 2 4 2 24 25
27 1 2 4 2 25 26
28 1 2 4 2 26 27
29 1 2 4 2 27 28
30 1 2 4 2 28 29
31 1 2 4 2 29 30
32 1 2 4 2 30 31
33 1 2 4 2 31 32
34 1 2 4 2 32 33
35 1 2 4 2 33 34
36 1 2 4 2 34 35
37 1 2 4 2 35 1
$EndElements
)";
  }

  return std::dynamic_pointer_cast<const Mesh<Connectivity<1>>>(GmshReader{filename}.mesh());
}

std::shared_ptr<const Mesh<Connectivity<2>>>
MeshDataBaseForTests::_buildHybrid2dMesh()
{
  const std::string filename = std::filesystem::path{PUGS_BINARY_DIR}.append("tests").append("hybrid-2d.msh");
  if (parallel::rank() == 0) {
    std::ofstream fout(filename);
    fout << R"($MeshFormat
2.2 0 8
$EndMeshFormat
$PhysicalNames
11
0 8 "XMINYMIN"
0 9 "XMINYMAX"
0 10 "XMAXYMIN"
0 11 "XMAXYMAX"
1 1 "XMIN"
1 2 "XMAX"
1 3 "YMAX"
1 4 "YMIN"
1 5 "INTERFACE"
2 6 "LEFT"
2 7 "RIGHT"
$EndPhysicalNames
$Nodes
53
1 0 0 0
2 1 0 0
3 1 1 0
4 0 1 0
5 2 0 0
6 2 1 0
7 0 0.7500000000003477 0
8 0 0.5000000000020616 0
9 0 0.2500000000010419 0
10 1 0.2499999999994109 0
11 1 0.4999999999986918 0
12 1 0.7499999999993401 0
13 2 0.2499999999994109 0
14 2 0.4999999999986918 0
15 2 0.7499999999993401 0
16 0.7500000000003477 1 0
17 0.5000000000020616 1 0
18 0.2500000000010419 1 0
19 1.749999999999999 1 0
20 1.499999999999998 1 0
21 1.249999999999999 1 0
22 0.2499999999994109 0 0
23 0.4999999999986918 0 0
24 0.7499999999993401 0 0
25 1.250000000000001 0 0
26 1.500000000000002 0 0
27 1.750000000000001 0 0
28 0.5645539210988633 0.7822119881665017 0
29 0.8034002623653069 0.616177001724073 0
30 0.6429040982169911 0.5281266166868597 0
31 0.8070137458488089 0.4050273873671746 0
32 0.193468206450602 0.7967828567280011 0
33 0.192644796000149 0.6419508693748935 0
34 0.3924123452244094 0.5739639975328588 0
35 0.3555609898784376 0.7915857937795373 0
36 0.1932938945983771 0.4291816994842411 0
37 0.3837729988924632 0.3582605556259788 0
38 0.5797710532625071 0.2819566416178859 0
39 0.7822395465040892 0.7780567773069014 0
40 0.355038784333282 0.2031847484281184 0
41 0.2059222030287144 0.2167595959294507 0
42 0.7859869002303115 0.2399319972346836 0
43 1.212848849190124 0.2732369527435796 0
44 1.307250451008527 0.5758879308566747 0
45 1.493594137086004 0.410605759034483 0
46 1.439097356236337 0.1761649793157971 0
47 1.574782862067863 0.7190230111887566 0
48 1.740502300977129 0.4934083027495074 0
49 1.853320217111459 0.2440179674328906 0
50 1.629419001691624 0.2206995018497341 0
51 1.844214193840832 0.743738552322716 0
52 1.151907331010239 0.7762099486172758 0
53 1.354193535052909 0.8313717608489495 0
$EndNodes
$Elements
90
1 15 2 8 1 1
2 15 2 9 4 4
3 15 2 10 5 5
4 15 2 11 6 6
5 1 2 1 1 4 7
6 1 2 1 1 7 8
7 1 2 1 1 8 9
8 1 2 1 1 9 1
9 1 2 5 2 2 10
10 1 2 5 2 10 11
11 1 2 5 2 11 12
12 1 2 5 2 12 3
13 1 2 2 3 5 13
14 1 2 2 3 13 14
15 1 2 2 3 14 15
16 1 2 2 3 15 6
17 1 2 3 4 3 16
18 1 2 3 4 16 17
19 1 2 3 4 17 18
20 1 2 3 4 18 4
21 1 2 3 5 6 19
22 1 2 3 5 19 20
23 1 2 3 5 20 21
24 1 2 3 5 21 3
25 1 2 4 6 1 22
26 1 2 4 6 22 23
27 1 2 4 6 23 24
28 1 2 4 6 24 2
29 1 2 4 7 2 25
30 1 2 4 7 25 26
31 1 2 4 7 26 27
32 1 2 4 7 27 5
33 2 2 7 2 43 45 44
34 2 2 7 2 46 45 43
35 2 2 7 2 11 43 44
36 2 2 7 2 45 48 47
37 2 2 7 2 46 43 25
38 2 2 7 2 44 45 47
39 2 2 7 2 43 11 10
40 2 2 7 2 48 50 49
41 2 2 7 2 14 51 48
42 2 2 7 2 45 50 48
43 2 2 7 2 27 49 50
44 2 2 7 2 52 11 44
45 2 2 7 2 49 27 5
46 2 2 7 2 47 19 20
47 2 2 7 2 12 52 3
48 2 2 7 2 52 21 3
49 2 2 7 2 47 53 44
50 2 2 7 2 27 50 26
51 2 2 7 2 43 10 2
52 2 2 7 2 25 43 2
53 2 2 7 2 20 53 47
54 2 2 7 2 21 53 20
55 2 2 7 2 46 50 45
56 2 2 7 2 26 50 46
57 2 2 7 2 44 53 52
58 2 2 7 2 46 25 26
59 2 2 7 2 11 52 12
60 2 2 7 2 47 48 51
61 2 2 7 2 51 14 15
62 2 2 7 2 13 49 5
63 2 2 7 2 51 19 47
64 2 2 7 2 21 52 53
65 2 2 7 2 19 51 6
66 2 2 7 2 51 15 6
67 2 2 7 2 48 49 14
68 2 2 7 2 49 13 14
69 3 2 6 1 37 34 33 36
70 3 2 6 1 30 31 11 29
71 3 2 6 1 35 34 30 28
72 3 2 6 1 30 29 39 28
73 3 2 6 1 34 37 38 30
74 3 2 6 1 37 40 23 38
75 3 2 6 1 41 40 37 36
76 3 2 6 1 28 17 18 35
77 3 2 6 1 30 38 42 31
78 3 2 6 1 11 31 42 10
79 3 2 6 1 39 16 17 28
80 3 2 6 1 33 32 4 7
81 3 2 6 1 11 12 39 29
82 3 2 6 1 35 18 4 32
83 3 2 6 1 23 40 41 22
84 3 2 6 1 16 39 12 3
85 3 2 6 1 41 36 8 9
86 3 2 6 1 9 1 22 41
87 3 2 6 1 10 42 24 2
88 3 2 6 1 23 24 42 38
89 3 2 6 1 7 8 36 33
90 3 2 6 1 35 32 33 34
$EndElements
)";
  }
  return std::dynamic_pointer_cast<const Mesh<Connectivity<2>>>(GmshReader{filename}.mesh());
}

std::shared_ptr<const Mesh<Connectivity<3>>>
MeshDataBaseForTests::_buildHybrid3dMesh()
{
  const std::string filename = std::filesystem::path{PUGS_BINARY_DIR}.append("tests").append("hybrid-3d.msh");
  if (parallel::rank() == 0) {
    std::ofstream fout(filename);
    fout << R"($MeshFormat
2.2 0 8
$EndMeshFormat
$PhysicalNames
31
0 40 "XMINYMINZMIN"
0 41 "XMAXYMINZMIN"
0 42 "XMINYMAXZMIN"
0 43 "XMINYMAXZMAX"
0 44 "XMINYMINZMAX"
0 45 "XMAXYMINZMAX"
0 47 "XMAXYMAXZMAX"
0 51 "XMAXYMAXZMIN"
1 28 "XMINZMIN"
1 29 "XMINZMAX"
1 30 "XMINYMAX"
1 31 "XMINYMIN"
1 32 "XMAXZMIN"
1 33 "XMAXZMAX"
1 34 "XMAXYMAX"
1 35 "XMAXYMIN"
1 36 "YMINZMIN"
1 37 "YMINZMAX"
1 38 "YMAXZMIN"
1 39 "YMAXZMAX"
2 22 "XMIN"
2 23 "XMAX"
2 24 "ZMAX"
2 25 "ZMIN"
2 26 "YMAX"
2 27 "YMIN"
2 55 "INTERFACE1"
2 56 "INTERFACE2"
3 52 "LEFT"
3 53 "MIDDLE"
3 54 "RIGHT"
$EndPhysicalNames
$Nodes
132
1 0 0 0
2 0.7 0 0
3 0.8 1 0
4 0 1 0
5 1.3 0 0
6 1.2 1 0
7 0 1 1
8 0 0 1
9 0.7 0 1
10 0.8 1 1
11 1.3 0 1
12 1.2 1 1
13 2 0 0
14 2 1 0
15 2 0 1
16 2 1 1
17 0 0.7500000000003466 0
18 0 0.5000000000020587 0
19 0 0.2500000000010403 0
20 0.7249999999999414 0.2499999999994139 0
21 0.7499999999998691 0.4999999999986909 0
22 0.7749999999999342 0.7499999999993421 0
23 1.266666666666749 0.3333333333325114 0
24 1.23333333333342 0.6666666666657952 0
25 0.4000000000010956 1 0
26 0.3499999999991014 0 0
27 0.9999999999987851 0 0
28 0 0.7500000000003466 1
29 0 0.5000000000020587 1
30 0 0.2500000000010403 1
31 0.3499999999991014 0 1
32 0.7249999999999414 0.2499999999994139 1
33 0.7499999999998691 0.4999999999986909 1
34 0.7749999999999342 0.7499999999993421 1
35 0.4000000000010956 1 1
36 0 1 0.3333333333333333
37 0 1 0.6666666666666666
38 0 0 0.3333333333333333
39 0 0 0.6666666666666666
40 0.7 0 0.3333333333333333
41 0.7 0 0.6666666666666666
42 0.8 1 0.3333333333333333
43 0.8 1 0.6666666666666666
44 0.9999999999987851 0 1
45 1.266666666666749 0.3333333333325114 1
46 1.23333333333342 0.6666666666657952 1
47 1.3 0 0.3333333333333333
48 1.3 0 0.6666666666666666
49 1.2 1 0.3333333333333333
50 1.2 1 0.6666666666666666
51 1.630495225600928 0 0
52 1.630495225600928 0 1
53 2 0 0.499999999998694
54 2 0.499999999998694 1
55 2 0.499999999998694 0
56 1.577708829260476 1 0
57 1.577708829258421 1 1
58 2 1 0.499999999998694
59 0.3962888297748171 0.7916234456658734 0
60 0.5588133104363274 0.5235006176096062 0
61 0.48540966431768 0.3326974099157888 0
62 0.2545886146233393 0.4410037988896774 0
63 0.1888904744336107 0.2469120833355397 0
64 0.9952483415423343 0.5591370310039079 0
65 1.009047204638942 0.8056163713004886 0
66 0.9496163434716166 0.3148023652713143 0
67 0 0.7500000000003466 0.3333333333333333
68 0 0.7500000000003466 0.6666666666666666
69 0 0.5000000000020587 0.3333333333333333
70 0 0.5000000000020587 0.6666666666666666
71 0 0.2500000000010403 0.3333333333333333
72 0 0.2500000000010403 0.6666666666666666
73 0.3499999999991014 0 0.3333333333333333
74 0.3499999999991014 0 0.6666666666666666
75 0.7249999999999414 0.2499999999994139 0.3333333333333333
76 0.7249999999999414 0.2499999999994139 0.6666666666666666
77 0.7499999999998691 0.4999999999986909 0.3333333333333333
78 0.7499999999998691 0.4999999999986909 0.6666666666666666
79 0.7749999999999342 0.7499999999993421 0.3333333333333333
80 0.7749999999999342 0.7499999999993421 0.6666666666666666
81 0.4000000000010956 1 0.3333333333333333
82 0.4000000000010956 1 0.6666666666666666
83 0.3962888297748171 0.7916234456658734 1
84 0.5588133104363274 0.5235006176096062 1
85 0.48540966431768 0.3326974099157888 1
86 0.2545886146233393 0.4410037988896774 1
87 0.1888904744336107 0.2469120833355397 1
88 0.9999999999987851 0 0.3333333333333333
89 0.9999999999987851 0 0.6666666666666666
90 1.266666666666749 0.3333333333325114 0.3333333333333333
91 1.266666666666749 0.3333333333325114 0.6666666666666666
92 1.23333333333342 0.6666666666657952 0.3333333333333333
93 1.23333333333342 0.6666666666657952 0.6666666666666666
94 0.9952483415423343 0.5591370310039079 1
95 1.009047204638942 0.8056163713004886 1
96 0.9496163434716166 0.3148023652713143 1
97 1.694794622046484 0 0.4999999999999999
98 2 0.667671095007814 0.3323289049917093
99 2 0.3400507564846686 0.5028063863717376
100 2 0.7126842003874787 0.6873157996117608
101 1.638498486684473 0.2509397074193687 1
102 1.619828855633658 0.6299359336032246 1
103 1.638498486684556 0.2509397074193682 0
104 1.619828855634071 0.629935933603235 0
105 1.657210007479593 1 0.5000000082593427
106 0.3962888297748171 0.7916234456658734 0.3333333333333333
107 0.3962888297748171 0.7916234456658734 0.6666666666666666
108 0.5588133104363274 0.5235006176096062 0.3333333333333333
109 0.5588133104363274 0.5235006176096062 0.6666666666666666
110 0.48540966431768 0.3326974099157888 0.3333333333333333
111 0.48540966431768 0.3326974099157888 0.6666666666666666
112 0.2545886146233393 0.4410037988896774 0.3333333333333333
113 0.2545886146233393 0.4410037988896774 0.6666666666666666
114 0.1888904744336107 0.2469120833355397 0.3333333333333333
115 0.1888904744336107 0.2469120833355397 0.6666666666666666
116 0.9952483415423343 0.5591370310039079 0.3333333333333333
117 0.9952483415423343 0.5591370310039079 0.6666666666666666
118 1.009047204638942 0.8056163713004886 0.3333333333333333
119 1.009047204638942 0.8056163713004886 0.6666666666666666
120 0.9496163434716166 0.3148023652713143 0.3333333333333333
121 0.9496163434716166 0.3148023652713143 0.6666666666666666
122 1.605048220195734 0.3680048220187183 0.5
123 1.521560099439846 0.6946560099431293 0.4999999999999999
124 1.449585918125941 0.1832919251455124 0.1666666666666667
125 1.449585918125941 0.1832919251455124 0.4999999999999999
126 1.449585918125941 0.1832919251455124 0.8333333333333333
127 1.416252584792843 0.5166252584784292 0.1666666666666667
128 1.416252584792843 0.5166252584784292 0.4999999999999999
129 1.416252584792843 0.5166252584784292 0.8333333333333333
130 1.382919251459699 0.8499585918121965 0.1666666666666667
131 1.382919251459699 0.8499585918121965 0.4999999999999999
132 1.382919251459699 0.8499585918121965 0.8333333333333333
$EndNodes
$Elements
405
1 15 2 40 1 1
2 15 2 42 4 4
3 15 2 43 7 7
4 15 2 44 8 8
5 15 2 41 27 13
6 15 2 51 28 14
7 15 2 45 29 15
8 15 2 47 30 16
9 1 2 28 1 4 17
10 1 2 28 1 17 18
11 1 2 28 1 18 19
12 1 2 28 1 19 1
13 1 2 38 4 3 25
14 1 2 38 4 25 4
15 1 2 38 5 6 3
16 1 2 36 6 1 26
17 1 2 36 6 26 2
18 1 2 36 7 2 27
19 1 2 36 7 27 5
20 1 2 29 12 7 28
21 1 2 29 12 28 29
22 1 2 29 12 29 30
23 1 2 29 12 30 8
24 1 2 37 13 8 31
25 1 2 37 13 31 9
26 1 2 39 15 10 35
27 1 2 39 15 35 7
28 1 2 30 17 4 36
29 1 2 30 17 36 37
30 1 2 30 17 37 7
31 1 2 31 18 1 38
32 1 2 31 18 38 39
33 1 2 31 18 39 8
34 1 2 37 35 9 44
35 1 2 37 35 44 11
36 1 2 39 37 12 10
37 1 2 36 49 5 51
38 1 2 36 49 51 13
39 1 2 37 50 11 52
40 1 2 37 50 52 15
41 1 2 35 51 13 53
42 1 2 35 51 53 15
43 1 2 33 52 15 54
44 1 2 33 52 54 16
45 1 2 32 53 13 55
46 1 2 32 53 55 14
47 1 2 38 54 14 56
48 1 2 38 54 56 6
49 1 2 39 55 12 57
50 1 2 39 55 57 16
51 1 2 34 56 14 58
52 1 2 34 56 58 16
53 2 2 25 2 20 2 27
54 2 2 25 2 23 27 5
55 2 2 25 2 64 23 24
56 2 2 25 2 22 65 3
57 2 2 25 2 23 64 66
58 2 2 25 2 65 6 3
59 2 2 25 2 64 21 66
60 2 2 25 2 65 24 6
61 2 2 25 2 24 65 64
62 2 2 25 2 65 22 64
63 2 2 25 2 66 20 27
64 2 2 25 2 21 64 22
65 2 2 25 2 20 66 21
66 2 2 25 2 23 66 27
67 2 2 24 54 32 9 44
68 2 2 24 54 45 44 11
69 2 2 24 54 94 45 46
70 2 2 24 54 34 95 10
71 2 2 24 54 45 94 96
72 2 2 24 54 95 12 10
73 2 2 24 54 94 33 96
74 2 2 24 54 95 46 12
75 2 2 24 54 46 95 94
76 2 2 24 54 95 34 94
77 2 2 24 54 96 32 44
78 2 2 24 54 33 94 34
79 2 2 24 54 32 96 33
80 2 2 24 54 45 96 44
81 2 2 27 55 51 5 47
82 2 2 27 55 48 11 52
83 2 2 27 55 51 97 13
84 2 2 27 55 97 53 13
85 2 2 27 55 97 52 15
86 2 2 27 55 15 53 97
87 2 2 27 55 97 47 48
88 2 2 27 55 47 97 51
89 2 2 27 55 48 52 97
90 2 2 23 56 53 99 13
91 2 2 23 56 99 55 13
92 2 2 23 56 55 98 14
93 2 2 23 56 98 58 14
94 2 2 23 56 99 53 15
95 2 2 23 56 54 99 15
96 2 2 23 56 100 54 16
97 2 2 23 56 58 100 16
98 2 2 23 56 100 99 54
99 2 2 23 56 55 99 98
100 2 2 23 56 100 58 98
101 2 2 23 56 98 99 100
102 2 2 24 57 45 101 11
103 2 2 24 57 101 52 11
104 2 2 24 57 46 12 57
105 2 2 24 57 52 101 15
106 2 2 24 57 101 54 15
107 2 2 24 57 54 102 16
108 2 2 24 57 102 57 16
109 2 2 24 57 46 102 45
110 2 2 24 57 101 45 102
111 2 2 24 57 102 46 57
112 2 2 24 57 102 54 101
113 2 2 25 58 23 103 5
114 2 2 25 58 103 51 5
115 2 2 25 58 24 6 56
116 2 2 25 58 51 103 13
117 2 2 25 58 103 55 13
118 2 2 25 58 55 104 14
119 2 2 25 58 104 56 14
120 2 2 25 58 24 104 23
121 2 2 25 58 103 23 104
122 2 2 25 58 104 24 56
123 2 2 25 58 104 55 103
124 2 2 26 59 6 49 56
125 2 2 26 59 50 12 57
126 2 2 26 59 56 105 14
127 2 2 26 59 105 58 14
128 2 2 26 59 105 57 16
129 2 2 26 59 16 58 105
130 2 2 26 59 105 49 50
131 2 2 26 59 49 105 56
132 2 2 26 59 50 57 105
133 3 2 25 1 59 60 21 22
134 3 2 25 1 22 3 25 59
135 3 2 25 1 18 62 59 17
136 3 2 25 1 25 4 17 59
137 3 2 25 1 62 63 26 61
138 3 2 25 1 60 61 20 21
139 3 2 25 1 62 18 19 63
140 3 2 25 1 26 2 20 61
141 3 2 25 1 61 60 59 62
142 3 2 25 1 19 1 26 63
143 3 2 22 19 4 17 67 36
144 3 2 22 19 36 67 68 37
145 3 2 22 19 37 68 28 7
146 3 2 22 19 17 18 69 67
147 3 2 22 19 67 69 70 68
148 3 2 22 19 68 70 29 28
149 3 2 22 19 18 19 71 69
150 3 2 22 19 69 71 72 70
151 3 2 22 19 70 72 30 29
152 3 2 22 19 19 1 38 71
153 3 2 22 19 71 38 39 72
154 3 2 22 19 72 39 8 30
155 3 2 27 23 1 26 73 38
156 3 2 27 23 38 73 74 39
157 3 2 27 23 39 74 31 8
158 3 2 27 23 26 2 40 73
159 3 2 27 23 73 40 41 74
160 3 2 27 23 74 41 9 31
161 3 2 55 27 2 20 75 40
162 3 2 55 27 40 75 76 41
163 3 2 55 27 41 76 32 9
164 3 2 55 27 20 21 77 75
165 3 2 55 27 75 77 78 76
166 3 2 55 27 76 78 33 32
167 3 2 55 27 21 22 79 77
168 3 2 55 27 77 79 80 78
169 3 2 55 27 78 80 34 33
170 3 2 55 27 22 3 42 79
171 3 2 55 27 79 42 43 80
172 3 2 55 27 80 43 10 34
173 3 2 26 31 3 25 81 42
174 3 2 26 31 42 81 82 43
175 3 2 26 31 43 82 35 10
176 3 2 26 31 25 4 36 81
177 3 2 26 31 81 36 37 82
178 3 2 26 31 82 37 7 35
179 3 2 24 32 83 84 33 34
180 3 2 24 32 34 10 35 83
181 3 2 24 32 29 86 83 28
182 3 2 24 32 35 7 28 83
183 3 2 24 32 86 87 31 85
184 3 2 24 32 84 85 32 33
185 3 2 24 32 86 29 30 87
186 3 2 24 32 31 9 32 85
187 3 2 24 32 85 84 83 86
188 3 2 24 32 30 8 31 87
189 3 2 27 45 2 27 88 40
190 3 2 27 45 40 88 89 41
191 3 2 27 45 41 89 44 9
192 3 2 27 45 27 5 47 88
193 3 2 27 45 88 47 48 89
194 3 2 27 45 89 48 11 44
195 3 2 56 49 5 23 90 47
196 3 2 56 49 47 90 91 48
197 3 2 56 49 48 91 45 11
198 3 2 56 49 23 24 92 90
199 3 2 56 49 90 92 93 91
200 3 2 56 49 91 93 46 45
201 3 2 56 49 24 6 49 92
202 3 2 56 49 92 49 50 93
203 3 2 56 49 93 50 12 46
204 3 2 26 53 6 3 42 49
205 3 2 26 53 49 42 43 50
206 3 2 26 53 50 43 10 12
207 4 2 54 3 90 103 127 122
208 4 2 54 3 101 91 129 122
209 4 2 54 3 103 90 124 122
210 4 2 54 3 126 91 101 122
211 4 2 54 3 92 104 130 123
212 4 2 54 3 102 93 132 123
213 4 2 54 3 129 45 91 101
214 4 2 54 3 90 23 127 103
215 4 2 54 3 45 126 91 101
216 4 2 54 3 23 90 124 103
217 4 2 54 3 92 127 104 123
218 4 2 54 3 93 102 129 123
219 4 2 54 3 125 90 91 122
220 4 2 54 3 91 90 128 122
221 4 2 54 3 104 127 103 122
222 4 2 54 3 101 129 102 122
223 4 2 54 3 127 122 104 123
224 4 2 54 3 102 122 129 123
225 4 2 54 3 103 124 97 122
226 4 2 54 3 126 101 97 122
227 4 2 54 3 132 57 102 123
228 4 2 54 3 130 104 56 123
229 4 2 54 3 46 93 132 102
230 4 2 54 3 92 24 130 104
231 4 2 54 3 129 93 46 102
232 4 2 54 3 92 127 24 104
233 4 2 54 3 102 100 54 122
234 4 2 54 3 100 122 102 123
235 4 2 54 3 54 101 102 122
236 4 2 54 3 104 103 55 122
237 4 2 54 3 51 103 124 97
238 4 2 54 3 126 101 52 97
239 4 2 54 3 105 130 56 123
240 4 2 54 3 132 105 57 123
241 4 2 54 3 98 104 55 122
242 4 2 54 3 93 92 131 123
243 4 2 54 3 93 128 92 123
244 4 2 54 3 51 124 47 97
245 4 2 54 3 52 48 126 97
246 4 2 54 3 54 100 99 122
247 4 2 54 3 104 122 98 123
248 4 2 54 3 98 122 100 123
249 4 2 54 3 54 99 101 122
250 4 2 54 3 55 103 99 122
251 4 2 54 3 49 130 56 105
252 4 2 54 3 57 132 50 105
253 4 2 54 3 55 99 98 122
254 4 2 54 3 103 99 97 53
255 4 2 54 3 101 15 97 53
256 4 2 54 3 97 99 101 53
257 4 2 54 3 105 100 102 123
258 4 2 54 3 53 97 103 13
259 4 2 54 3 103 97 99 122
260 4 2 54 3 53 103 99 13
261 4 2 54 3 15 101 99 53
262 4 2 54 3 99 97 101 122
263 4 2 54 3 105 104 98 123
264 4 2 54 3 105 57 102 100
265 4 2 54 3 98 105 56 104
266 4 2 54 3 12 132 50 57
267 4 2 54 3 56 6 130 49
268 4 2 54 3 104 127 23 103
269 4 2 54 3 129 45 101 102
270 4 2 54 3 103 55 99 13
271 4 2 54 3 15 99 101 54
272 4 2 54 3 14 98 56 104
273 4 2 54 3 52 101 15 97
274 4 2 54 3 97 51 103 13
275 4 2 54 3 105 56 104 123
276 4 2 54 3 57 105 102 123
277 4 2 54 3 56 105 98 14
278 4 2 54 3 100 98 99 122
279 4 2 54 3 57 12 132 46
280 4 2 54 3 24 56 6 130
281 4 2 54 3 105 16 57 100
282 4 2 54 3 16 102 57 100
283 4 2 54 3 126 11 48 52
284 4 2 54 3 5 124 47 51
285 4 2 54 3 58 98 105 14
286 4 2 54 3 132 57 46 102
287 4 2 54 3 24 56 130 104
288 4 2 54 3 105 98 100 123
289 4 2 54 3 100 105 16 58
290 4 2 54 3 11 126 45 101
291 4 2 54 3 23 124 5 103
292 4 2 54 3 50 131 49 105
293 4 2 54 3 45 129 46 102
294 4 2 54 3 24 127 23 104
295 4 2 54 3 54 102 16 100
296 4 2 54 3 104 14 98 55
297 4 2 54 3 124 5 103 51
298 4 2 54 3 11 126 101 52
299 4 2 54 3 105 100 98 58
300 4 2 54 3 47 125 48 97
301 4 2 54 3 93 128 129 91
302 4 2 54 3 93 129 128 123
303 4 2 54 3 122 129 128 91
304 4 2 54 3 122 128 129 123
305 4 2 54 3 122 128 127 90
306 4 2 54 3 122 127 128 123
307 4 2 54 3 92 127 128 90
308 4 2 54 3 92 128 127 123
309 4 2 54 3 91 125 126 48
310 4 2 54 3 91 126 125 122
311 4 2 54 3 97 126 125 48
312 4 2 54 3 97 125 126 122
313 4 2 54 3 97 125 124 47
314 4 2 54 3 97 124 125 122
315 4 2 54 3 90 124 125 47
316 4 2 54 3 90 125 124 122
317 4 2 54 3 105 131 132 50
318 4 2 54 3 105 132 131 123
319 4 2 54 3 93 132 131 50
320 4 2 54 3 93 131 132 123
321 4 2 54 3 92 131 130 49
322 4 2 54 3 92 130 131 123
323 4 2 54 3 105 130 131 49
324 4 2 54 3 105 131 130 123
325 5 2 52 1 21 22 59 60 77 79 106 108
326 5 2 52 1 77 79 106 108 78 80 107 109
327 5 2 52 1 78 80 107 109 33 34 83 84
328 5 2 52 1 25 59 22 3 81 106 79 42
329 5 2 52 1 81 106 79 42 82 107 80 43
330 5 2 52 1 82 107 80 43 35 83 34 10
331 5 2 52 1 59 17 18 62 106 67 69 112
332 5 2 52 1 106 67 69 112 107 68 70 113
333 5 2 52 1 107 68 70 113 83 28 29 86
334 5 2 52 1 17 59 25 4 67 106 81 36
335 5 2 52 1 67 106 81 36 68 107 82 37
336 5 2 52 1 68 107 82 37 28 83 35 7
337 5 2 52 1 26 61 62 63 73 110 112 114
338 5 2 52 1 73 110 112 114 74 111 113 115
339 5 2 52 1 74 111 113 115 31 85 86 87
340 5 2 52 1 20 21 60 61 75 77 108 110
341 5 2 52 1 75 77 108 110 76 78 109 111
342 5 2 52 1 76 78 109 111 32 33 84 85
343 5 2 52 1 19 63 62 18 71 114 112 69
344 5 2 52 1 71 114 112 69 72 115 113 70
345 5 2 52 1 72 115 113 70 30 87 86 29
346 5 2 52 1 20 61 26 2 75 110 73 40
347 5 2 52 1 75 110 73 40 76 111 74 41
348 5 2 52 1 76 111 74 41 32 85 31 9
349 5 2 52 1 59 62 61 60 106 112 110 108
350 5 2 52 1 106 112 110 108 107 113 111 109
351 5 2 52 1 107 113 111 109 83 86 85 84
352 5 2 52 1 26 63 19 1 73 114 71 38
353 5 2 52 1 73 114 71 38 74 115 72 39
354 5 2 52 1 74 115 72 39 31 87 30 8
355 6 2 53 2 27 20 2 88 75 40
356 6 2 53 2 88 75 40 89 76 41
357 6 2 53 2 89 76 41 44 32 9
358 6 2 53 2 5 23 27 47 90 88
359 6 2 53 2 47 90 88 48 91 89
360 6 2 53 2 48 91 89 11 45 44
361 6 2 53 2 24 64 23 92 116 90
362 6 2 53 2 92 116 90 93 117 91
363 6 2 53 2 93 117 91 46 94 45
364 6 2 53 2 3 22 65 42 79 118
365 6 2 53 2 42 79 118 43 80 119
366 6 2 53 2 43 80 119 10 34 95
367 6 2 53 2 66 23 64 120 90 116
368 6 2 53 2 120 90 116 121 91 117
369 6 2 53 2 121 91 117 96 45 94
370 6 2 53 2 3 65 6 42 118 49
371 6 2 53 2 42 118 49 43 119 50
372 6 2 53 2 43 119 50 10 95 12
373 6 2 53 2 66 64 21 120 116 77
374 6 2 53 2 120 116 77 121 117 78
375 6 2 53 2 121 117 78 96 94 33
376 6 2 53 2 6 65 24 49 118 92
377 6 2 53 2 49 118 92 50 119 93
378 6 2 53 2 50 119 93 12 95 46
379 6 2 53 2 64 24 65 116 92 118
380 6 2 53 2 116 92 118 117 93 119
381 6 2 53 2 117 93 119 94 46 95
382 6 2 53 2 64 65 22 116 118 79
383 6 2 53 2 116 118 79 117 119 80
384 6 2 53 2 117 119 80 94 95 34
385 6 2 53 2 27 66 20 88 120 75
386 6 2 53 2 88 120 75 89 121 76
387 6 2 53 2 89 121 76 44 96 32
388 6 2 53 2 22 21 64 79 77 116
389 6 2 53 2 79 77 116 80 78 117
390 6 2 53 2 80 78 117 34 33 94
391 6 2 53 2 21 20 66 77 75 120
392 6 2 53 2 77 75 120 78 76 121
393 6 2 53 2 78 76 121 33 32 96
394 6 2 53 2 27 23 66 88 90 120
395 6 2 53 2 88 90 120 89 91 121
396 6 2 53 2 89 91 121 44 45 96
397 7 2 54 3 5 23 90 47 124
398 7 2 54 3 47 90 91 48 125
399 7 2 54 3 48 91 45 11 126
400 7 2 54 3 23 24 92 90 127
401 7 2 54 3 90 92 93 91 128
402 7 2 54 3 91 93 46 45 129
403 7 2 54 3 24 6 49 92 130
404 7 2 54 3 92 49 50 93 131
405 7 2 54 3 93 50 12 46 132
$EndElements
$Periodic
9
0 7 4
1
7 4
0 8 1
1
8 1
1 12 1
3
28 17
29 18
30 19
1 13 6
1
31 26
1 15 4
1
35 25
1 35 7
1
44 27
1 37 5
0
2 32 1
5
86 62
87 63
83 59
84 60
85 61
2 54 2
3
94 64
95 65
96 66
$EndPeriodic
)";
  }
  return std::dynamic_pointer_cast<const Mesh<Connectivity<3>>>(GmshReader{filename}.mesh());
}
