#ifndef VTK_WRITER_HPP
#define VTK_WRITER_HPP

#include <output/WriterBase.hpp>

#include <algebra/TinyMatrix.hpp>
#include <algebra/TinyVector.hpp>
#include <output/OutputNamedItemValueSet.hpp>

class IMesh;

#include <string>

class VTKWriter : public WriterBase
{
 private:
  class SerializedDataList;

  std::string _getDateAndVersionComment() const;

  std::string _getFilenamePVTU() const;

  std::string _getFilenameVTU(int rank_number) const;

  template <typename ItemDataT>
  void _write_item_pvtu(std::ofstream& os, const std::string& name, const ItemDataT&) const;

  template <typename ItemDataT>
  void _write_node_pvtu(std::ofstream& os, const std::string& name, const ItemDataT&) const;

  template <typename ItemDataT>
  void _write_cell_pvtu(std::ofstream& os, const std::string& name, const ItemDataT&) const;

  template <typename DataType>
  struct VTKType;

  template <typename DataType>
  void _write_array(std::ofstream& os,
                    const std::string& name,
                    const Array<DataType>& array,
                    SerializedDataList& serialized_data_list) const;

  template <size_t N, typename DataType>
  void _write_array(std::ofstream& os,
                    const std::string& name,
                    const Array<TinyVector<N, DataType>>& array,
                    SerializedDataList& serialized_data_list) const;

  template <typename DataType>
  void _write_node_value(std::ofstream& os,
                         const std::string& name,
                         const NodeValue<const DataType>& item_value,
                         SerializedDataList& serialized_data_list) const;

  template <size_t N, typename DataType>
  void _write_node_value(std::ofstream& os,
                         const std::string& name,
                         const NodeValue<const TinyVector<N, DataType>>& item_value,
                         SerializedDataList& serialized_data_list) const;

  template <size_t N, typename DataType>
  void _write_node_value(std::ofstream& os,
                         const std::string& name,
                         const NodeValue<const TinyMatrix<N, N, DataType>>& item_value,
                         SerializedDataList& serialized_data_list) const;

  template <typename DataType>
  void _write_node_value(std::ofstream&,
                         const std::string&,
                         const CellValue<const DataType>&,
                         SerializedDataList&) const;

  template <typename DataType>
  void _write_cell_value(std::ofstream& os,
                         const std::string& name,
                         const CellValue<const DataType>& item_value,
                         SerializedDataList& serialized_data_list) const;

  template <typename ItemDataT>
  void _write_item_data(std::ofstream& os,
                        const std::string& name,
                        const ItemDataT& item_data,
                        SerializedDataList& serialized_data_list) const;

  template <typename ItemDataT>
  void _write_cell_data(std::ofstream& os,
                        const std::string& name,
                        const ItemDataT& item_data,
                        SerializedDataList& serialized_data_list) const;

  template <typename ItemDataT>
  void _write_node_data(std::ofstream& os,
                        const std::string& name,
                        const ItemDataT& item_data,
                        SerializedDataList& serialized_data_list) const;

  template <typename MeshType>
  void _write(const std::shared_ptr<const MeshType>& mesh,
              const OutputNamedItemDataSet& output_named_item_data_set,
              double time) const;

 public:
  void writeMesh(const std::shared_ptr<const IMesh>& mesh) const final;

  void write(const std::vector<std::shared_ptr<const NamedDiscreteFunction>>& named_discrete_function_list,
             double time) const final;

  VTKWriter(const std::string& base_filename, const double time_period) : WriterBase(base_filename, time_period) {}

  ~VTKWriter() = default;
};

#endif   // VTK_WRITER_HPP
