#include <mesh/GmshReader.hpp>

#include <utils/PugsMacros.hpp>

#include <mesh/CellType.hpp>
#include <mesh/Connectivity.hpp>
#include <mesh/ConnectivityBuilderBase.hpp>
#include <mesh/ConnectivityDispatcher.hpp>
#include <mesh/ItemValueUtils.hpp>
#include <mesh/Mesh.hpp>
#include <mesh/RefItemList.hpp>
#include <utils/Exceptions.hpp>

#include <rang.hpp>

#include <fstream>
#include <iomanip>
#include <iostream>
#include <map>
#include <regex>
#include <set>
#include <sstream>
#include <unordered_map>

template <size_t Dimension>
class GmshConnectivityBuilder : public ConnectivityBuilderBase
{
 public:
  GmshConnectivityBuilder(const GmshReader::GmshData& gmsh_data, const size_t nb_cells);
};

template <>
GmshConnectivityBuilder<1>::GmshConnectivityBuilder(const GmshReader::GmshData& gmsh_data, const size_t nb_cells)
{
  ConnectivityDescriptor descriptor;

  descriptor.node_number_vector = gmsh_data.__verticesNumbers;
  descriptor.cell_type_vector.resize(nb_cells);
  descriptor.cell_number_vector.resize(nb_cells);
  descriptor.cell_to_node_vector.resize(nb_cells);

  for (size_t j = 0; j < nb_cells; ++j) {
    descriptor.cell_to_node_vector[j].resize(2);
    for (int r = 0; r < 2; ++r) {
      descriptor.cell_to_node_vector[j][r] = gmsh_data.__edges[j][r];
    }
    descriptor.cell_type_vector[j]   = CellType::Line;
    descriptor.cell_number_vector[j] = gmsh_data.__edges_number[j];
  }

  std::map<unsigned int, std::vector<unsigned int>> ref_points_map;
  for (unsigned int r = 0; r < gmsh_data.__points.size(); ++r) {
    const unsigned int point_number = gmsh_data.__points[r];
    const unsigned int& ref         = gmsh_data.__points_ref[r];
    ref_points_map[ref].push_back(point_number);
  }

  for (const auto& ref_point_list : ref_points_map) {
    Array<NodeId> point_list(ref_point_list.second.size());
    for (size_t j = 0; j < ref_point_list.second.size(); ++j) {
      point_list[j] = ref_point_list.second[j];
    }
    const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_point_list.first);
    descriptor.addRefItemList(RefNodeList(physical_ref_id.refId(), point_list));
  }

  std::map<unsigned int, std::vector<unsigned int>> ref_cells_map;
  for (unsigned int j = 0; j < gmsh_data.__edges_ref.size(); ++j) {
    const unsigned int elem_number = j;
    const unsigned int& ref        = gmsh_data.__edges_ref[j];
    ref_cells_map[ref].push_back(elem_number);
  }

  for (const auto& ref_cell_list : ref_cells_map) {
    Array<CellId> cell_list(ref_cell_list.second.size());
    for (size_t j = 0; j < ref_cell_list.second.size(); ++j) {
      cell_list[j] = ref_cell_list.second[j];
    }
    const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_cell_list.first);
    descriptor.addRefItemList(RefCellList(physical_ref_id.refId(), cell_list));
  }

  descriptor.cell_owner_vector.resize(nb_cells);
  std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

  descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
  std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

  m_connectivity = Connectivity1D::build(descriptor);
}

template <>
GmshConnectivityBuilder<2>::GmshConnectivityBuilder(const GmshReader::GmshData& gmsh_data, const size_t nb_cells)
{
  ConnectivityDescriptor descriptor;

  descriptor.node_number_vector = gmsh_data.__verticesNumbers;
  descriptor.cell_type_vector.resize(nb_cells);
  descriptor.cell_number_vector.resize(nb_cells);
  descriptor.cell_to_node_vector.resize(nb_cells);

  const size_t nb_triangles = gmsh_data.__triangles.size();
  for (size_t j = 0; j < nb_triangles; ++j) {
    descriptor.cell_to_node_vector[j].resize(3);
    for (int r = 0; r < 3; ++r) {
      descriptor.cell_to_node_vector[j][r] = gmsh_data.__triangles[j][r];
    }
    descriptor.cell_type_vector[j]   = CellType::Triangle;
    descriptor.cell_number_vector[j] = gmsh_data.__triangles_number[j];
  }

  const size_t nb_quadrangles = gmsh_data.__quadrangles.size();
  for (size_t j = 0; j < nb_quadrangles; ++j) {
    const size_t jq = j + nb_triangles;
    descriptor.cell_to_node_vector[jq].resize(4);
    for (int r = 0; r < 4; ++r) {
      descriptor.cell_to_node_vector[jq][r] = gmsh_data.__quadrangles[j][r];
    }
    descriptor.cell_type_vector[jq]   = CellType::Quadrangle;
    descriptor.cell_number_vector[jq] = gmsh_data.__quadrangles_number[j];
  }

  std::map<unsigned int, std::vector<unsigned int>> ref_cells_map;
  for (unsigned int j = 0; j < gmsh_data.__triangles_ref.size(); ++j) {
    const unsigned int elem_number = j;
    const unsigned int& ref        = gmsh_data.__triangles_ref[j];
    ref_cells_map[ref].push_back(elem_number);
  }

  for (unsigned int j = 0; j < gmsh_data.__quadrangles_ref.size(); ++j) {
    const size_t elem_number = nb_triangles + j;
    const unsigned int& ref  = gmsh_data.__quadrangles_ref[j];
    ref_cells_map[ref].push_back(elem_number);
  }

  for (const auto& ref_cell_list : ref_cells_map) {
    Array<CellId> cell_list(ref_cell_list.second.size());
    for (size_t j = 0; j < ref_cell_list.second.size(); ++j) {
      cell_list[j] = ref_cell_list.second[j];
    }
    const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_cell_list.first);
    descriptor.addRefItemList(RefCellList(physical_ref_id.refId(), cell_list));
  }

  ConnectivityBuilderBase::_computeCellFaceAndFaceNodeConnectivities<2>(descriptor);

  using Face                                                                 = ConnectivityFace<2>;
  const auto& node_number_vector                                             = descriptor.node_number_vector;
  const std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map = [&] {
    std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map;
    for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
      const auto& node_vector                               = descriptor.face_to_node_vector[l];
      face_to_id_map[Face(node_vector, node_number_vector)] = l;
    }
    return face_to_id_map;
  }();

  std::unordered_map<int, FaceId> face_number_id_map = [&] {
    std::unordered_map<int, FaceId> face_number_id_map;
    for (size_t l = 0; l < descriptor.face_number_vector.size(); ++l) {
      face_number_id_map[descriptor.face_number_vector[l]] = l;
    }
    Assert(face_number_id_map.size() == descriptor.face_number_vector.size());
    return face_number_id_map;
  }();

  std::map<unsigned int, std::vector<unsigned int>> ref_faces_map;
  for (unsigned int e = 0; e < gmsh_data.__edges.size(); ++e) {
    const unsigned int edge_id = [&] {
      auto i = face_to_id_map.find(Face({gmsh_data.__edges[e][0], gmsh_data.__edges[e][1]}, node_number_vector));
      if (i == face_to_id_map.end()) {
        std::stringstream error_msg;
        error_msg << "face " << gmsh_data.__edges[e][0] << " not found";
        throw NormalError(error_msg.str());
      }
      return i->second;
    }();
    const unsigned int& ref = gmsh_data.__edges_ref[e];
    ref_faces_map[ref].push_back(edge_id);

    if (descriptor.face_number_vector[edge_id] != gmsh_data.__edges_number[e]) {
      if (auto i_face = face_number_id_map.find(gmsh_data.__edges_number[e]); i_face != face_number_id_map.end()) {
        const int other_edge_id = i_face->second;
        std::swap(descriptor.face_number_vector[edge_id], descriptor.face_number_vector[other_edge_id]);

        face_number_id_map.erase(descriptor.face_number_vector[edge_id]);
        face_number_id_map.erase(descriptor.face_number_vector[other_edge_id]);

        face_number_id_map[descriptor.face_number_vector[edge_id]]       = edge_id;
        face_number_id_map[descriptor.face_number_vector[other_edge_id]] = other_edge_id;
      } else {
        face_number_id_map.erase(descriptor.face_number_vector[edge_id]);
        descriptor.face_number_vector[edge_id]                     = gmsh_data.__edges_number[e];
        face_number_id_map[descriptor.face_number_vector[edge_id]] = edge_id;
      }
    }
  }

  for (const auto& ref_face_list : ref_faces_map) {
    Array<FaceId> face_list(ref_face_list.second.size());
    for (size_t j = 0; j < ref_face_list.second.size(); ++j) {
      face_list[j] = ref_face_list.second[j];
    }
    const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_face_list.first);
    descriptor.addRefItemList(RefFaceList{physical_ref_id.refId(), face_list});
  }

  std::map<unsigned int, std::vector<unsigned int>> ref_points_map;
  for (unsigned int r = 0; r < gmsh_data.__points.size(); ++r) {
    const unsigned int point_number = gmsh_data.__points[r];
    const unsigned int& ref         = gmsh_data.__points_ref[r];
    ref_points_map[ref].push_back(point_number);
  }

  for (const auto& ref_point_list : ref_points_map) {
    Array<NodeId> point_list(ref_point_list.second.size());
    for (size_t j = 0; j < ref_point_list.second.size(); ++j) {
      point_list[j] = ref_point_list.second[j];
    }
    const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_point_list.first);
    descriptor.addRefItemList(RefNodeList(physical_ref_id.refId(), point_list));
  }

  descriptor.cell_owner_vector.resize(nb_cells);
  std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

  descriptor.face_owner_vector.resize(descriptor.face_number_vector.size());
  std::fill(descriptor.face_owner_vector.begin(), descriptor.face_owner_vector.end(), parallel::rank());

  descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
  std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

  m_connectivity = Connectivity2D::build(descriptor);
}

template <>
GmshConnectivityBuilder<3>::GmshConnectivityBuilder(const GmshReader::GmshData& gmsh_data, const size_t nb_cells)
{
  ConnectivityDescriptor descriptor;

  descriptor.node_number_vector = gmsh_data.__verticesNumbers;
  descriptor.cell_type_vector.resize(nb_cells);
  descriptor.cell_number_vector.resize(nb_cells);
  descriptor.cell_to_node_vector.resize(nb_cells);

  const size_t nb_tetrahedra = gmsh_data.__tetrahedra.size();
  for (size_t j = 0; j < nb_tetrahedra; ++j) {
    descriptor.cell_to_node_vector[j].resize(4);
    for (int r = 0; r < 4; ++r) {
      descriptor.cell_to_node_vector[j][r] = gmsh_data.__tetrahedra[j][r];
    }
    descriptor.cell_type_vector[j]   = CellType::Tetrahedron;
    descriptor.cell_number_vector[j] = gmsh_data.__tetrahedra_number[j];
  }

  const size_t nb_hexahedra = gmsh_data.__hexahedra.size();
  for (size_t j = 0; j < nb_hexahedra; ++j) {
    const size_t jh = nb_tetrahedra + j;
    descriptor.cell_to_node_vector[jh].resize(8);
    for (int r = 0; r < 8; ++r) {
      descriptor.cell_to_node_vector[jh][r] = gmsh_data.__hexahedra[j][r];
    }
    descriptor.cell_type_vector[jh]   = CellType::Hexahedron;
    descriptor.cell_number_vector[jh] = gmsh_data.__hexahedra_number[j];
  }

  const size_t nb_prisms = gmsh_data.__prisms.size();
  for (size_t j = 0; j < nb_prisms; ++j) {
    const size_t jp = nb_tetrahedra + nb_hexahedra + j;
    descriptor.cell_to_node_vector[jp].resize(6);
    for (int r = 0; r < 6; ++r) {
      descriptor.cell_to_node_vector[jp][r] = gmsh_data.__prisms[j][r];
    }
    descriptor.cell_type_vector[jp]   = CellType::Prism;
    descriptor.cell_number_vector[jp] = gmsh_data.__prisms_number[j];
  }

  const size_t nb_pyramids = gmsh_data.__pyramids.size();
  for (size_t j = 0; j < nb_pyramids; ++j) {
    const size_t jh = nb_tetrahedra + nb_hexahedra + nb_prisms + j;
    descriptor.cell_to_node_vector[jh].resize(5);
    for (int r = 0; r < 5; ++r) {
      descriptor.cell_to_node_vector[jh][r] = gmsh_data.__pyramids[j][r];
    }
    descriptor.cell_type_vector[jh]   = CellType::Pyramid;
    descriptor.cell_number_vector[jh] = gmsh_data.__pyramids_number[j];
  }

  std::map<unsigned int, std::vector<unsigned int>> ref_cells_map;
  for (unsigned int j = 0; j < gmsh_data.__tetrahedra_ref.size(); ++j) {
    const unsigned int elem_number = j;
    const unsigned int& ref        = gmsh_data.__tetrahedra_ref[j];
    ref_cells_map[ref].push_back(elem_number);
  }

  for (unsigned int j = 0; j < gmsh_data.__hexahedra_ref.size(); ++j) {
    const size_t elem_number = nb_tetrahedra + j;
    const unsigned int& ref  = gmsh_data.__hexahedra_ref[j];
    ref_cells_map[ref].push_back(elem_number);
  }

  for (unsigned int j = 0; j < gmsh_data.__prisms_ref.size(); ++j) {
    const size_t elem_number = nb_tetrahedra + nb_hexahedra + j;
    const unsigned int& ref  = gmsh_data.__prisms_ref[j];
    ref_cells_map[ref].push_back(elem_number);
  }

  for (unsigned int j = 0; j < gmsh_data.__pyramids_ref.size(); ++j) {
    const size_t elem_number = nb_tetrahedra + nb_hexahedra + nb_prisms + j;
    const unsigned int& ref  = gmsh_data.__pyramids_ref[j];
    ref_cells_map[ref].push_back(elem_number);
  }

  for (const auto& ref_cell_list : ref_cells_map) {
    Array<CellId> cell_list(ref_cell_list.second.size());
    for (size_t j = 0; j < ref_cell_list.second.size(); ++j) {
      cell_list[j] = ref_cell_list.second[j];
    }
    const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_cell_list.first);
    descriptor.addRefItemList(RefCellList(physical_ref_id.refId(), cell_list));
  }

  ConnectivityBuilderBase::_computeCellFaceAndFaceNodeConnectivities<3>(descriptor);

  const auto& node_number_vector = descriptor.node_number_vector;

  {
    using Face                                                                 = ConnectivityFace<3>;
    const std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map = [&] {
      std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map;
      for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
        const auto& node_vector                               = descriptor.face_to_node_vector[l];
        face_to_id_map[Face(node_vector, node_number_vector)] = l;
      }
      return face_to_id_map;
    }();

    std::unordered_map<int, FaceId> face_number_id_map = [&] {
      std::unordered_map<int, FaceId> face_number_id_map;
      for (size_t l = 0; l < descriptor.face_number_vector.size(); ++l) {
        face_number_id_map[descriptor.face_number_vector[l]] = l;
      }
      Assert(face_number_id_map.size() == descriptor.face_number_vector.size());
      return face_number_id_map;
    }();

    std::map<unsigned int, std::vector<unsigned int>> ref_faces_map;
    for (unsigned int f = 0; f < gmsh_data.__triangles.size(); ++f) {
      const unsigned int face_id = [&] {
        auto i = face_to_id_map.find(
          Face({gmsh_data.__triangles[f][0], gmsh_data.__triangles[f][1], gmsh_data.__triangles[f][2]},
               node_number_vector));
        if (i == face_to_id_map.end()) {
          throw NormalError("face not found");
        }
        return i->second;
      }();

      const unsigned int& ref = gmsh_data.__triangles_ref[f];
      ref_faces_map[ref].push_back(face_id);

      if (descriptor.face_number_vector[face_id] != gmsh_data.__triangles_number[f]) {
        if (auto i_face = face_number_id_map.find(gmsh_data.__triangles_number[f]);
            i_face != face_number_id_map.end()) {
          const int other_face_id = i_face->second;
          std::swap(descriptor.face_number_vector[face_id], descriptor.face_number_vector[other_face_id]);

          face_number_id_map.erase(descriptor.face_number_vector[face_id]);
          face_number_id_map.erase(descriptor.face_number_vector[other_face_id]);

          face_number_id_map[descriptor.face_number_vector[face_id]]       = face_id;
          face_number_id_map[descriptor.face_number_vector[other_face_id]] = other_face_id;
        } else {
          face_number_id_map.erase(descriptor.face_number_vector[face_id]);
          descriptor.face_number_vector[face_id]                     = gmsh_data.__triangles_number[f];
          face_number_id_map[descriptor.face_number_vector[face_id]] = face_id;
        }
      }
    }

    for (unsigned int f = 0; f < gmsh_data.__quadrangles.size(); ++f) {
      const unsigned int face_id = [&] {
        auto i = face_to_id_map.find(Face({gmsh_data.__quadrangles[f][0], gmsh_data.__quadrangles[f][1],
                                           gmsh_data.__quadrangles[f][2], gmsh_data.__quadrangles[f][3]},
                                          node_number_vector));
        if (i == face_to_id_map.end()) {
          throw NormalError("face not found");
        }
        return i->second;
      }();

      const unsigned int& ref = gmsh_data.__quadrangles_ref[f];
      ref_faces_map[ref].push_back(face_id);

      if (descriptor.face_number_vector[face_id] != gmsh_data.__quadrangles_number[f]) {
        if (auto i_face = face_number_id_map.find(gmsh_data.__quadrangles_number[f]);
            i_face != face_number_id_map.end()) {
          const int other_face_id = i_face->second;
          std::swap(descriptor.face_number_vector[face_id], descriptor.face_number_vector[other_face_id]);

          face_number_id_map.erase(descriptor.face_number_vector[face_id]);
          face_number_id_map.erase(descriptor.face_number_vector[other_face_id]);

          face_number_id_map[descriptor.face_number_vector[face_id]]       = face_id;
          face_number_id_map[descriptor.face_number_vector[other_face_id]] = other_face_id;
        } else {
          face_number_id_map.erase(descriptor.face_number_vector[face_id]);
          descriptor.face_number_vector[face_id]                     = gmsh_data.__quadrangles_number[f];
          face_number_id_map[descriptor.face_number_vector[face_id]] = face_id;
        }
      }
    }

    for (const auto& ref_face_list : ref_faces_map) {
      Array<FaceId> face_list(ref_face_list.second.size());
      for (size_t j = 0; j < ref_face_list.second.size(); ++j) {
        face_list[j] = ref_face_list.second[j];
      }
      const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_face_list.first);
      descriptor.addRefItemList(RefFaceList{physical_ref_id.refId(), face_list});
    }
  }

  ConnectivityBuilderBase::_computeFaceEdgeAndEdgeNodeAndCellEdgeConnectivities<3>(descriptor);

  {
    using Edge                                                                 = ConnectivityFace<2>;
    const auto& node_number_vector                                             = descriptor.node_number_vector;
    const std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map = [&] {
      std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map;
      for (EdgeId l = 0; l < descriptor.edge_to_node_vector.size(); ++l) {
        const auto& node_vector                               = descriptor.edge_to_node_vector[l];
        edge_to_id_map[Edge(node_vector, node_number_vector)] = l;
      }
      return edge_to_id_map;
    }();

    std::unordered_map<int, EdgeId> edge_number_id_map = [&] {
      std::unordered_map<int, EdgeId> edge_number_id_map;
      for (size_t l = 0; l < descriptor.edge_number_vector.size(); ++l) {
        edge_number_id_map[descriptor.edge_number_vector[l]] = l;
      }
      Assert(edge_number_id_map.size() == descriptor.edge_number_vector.size());
      return edge_number_id_map;
    }();

    std::map<unsigned int, std::vector<unsigned int>> ref_edges_map;
    for (unsigned int e = 0; e < gmsh_data.__edges.size(); ++e) {
      const unsigned int edge_id = [&] {
        auto i = edge_to_id_map.find(Edge({gmsh_data.__edges[e][0], gmsh_data.__edges[e][1]}, node_number_vector));
        if (i == edge_to_id_map.end()) {
          std::stringstream error_msg;
          error_msg << "edge " << gmsh_data.__edges[e][0] << " not found";
          throw NormalError(error_msg.str());
        }
        return i->second;
      }();
      const unsigned int& ref = gmsh_data.__edges_ref[e];
      ref_edges_map[ref].push_back(edge_id);

      if (descriptor.edge_number_vector[edge_id] != gmsh_data.__edges_number[e]) {
        if (auto i_edge = edge_number_id_map.find(gmsh_data.__edges_number[e]); i_edge != edge_number_id_map.end()) {
          const int other_edge_id = i_edge->second;
          std::swap(descriptor.edge_number_vector[edge_id], descriptor.edge_number_vector[other_edge_id]);

          edge_number_id_map.erase(descriptor.edge_number_vector[edge_id]);
          edge_number_id_map.erase(descriptor.edge_number_vector[other_edge_id]);

          edge_number_id_map[descriptor.edge_number_vector[edge_id]]       = edge_id;
          edge_number_id_map[descriptor.edge_number_vector[other_edge_id]] = other_edge_id;
        } else {
          edge_number_id_map.erase(descriptor.edge_number_vector[edge_id]);
          descriptor.edge_number_vector[edge_id]                     = gmsh_data.__edges_number[e];
          edge_number_id_map[descriptor.edge_number_vector[edge_id]] = edge_id;
        }
      }
    }

    for (const auto& ref_edge_list : ref_edges_map) {
      Array<EdgeId> edge_list(ref_edge_list.second.size());
      for (size_t j = 0; j < ref_edge_list.second.size(); ++j) {
        edge_list[j] = ref_edge_list.second[j];
      }
      const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_edge_list.first);
      descriptor.addRefItemList(RefEdgeList{physical_ref_id.refId(), edge_list});
    }
  }

  std::map<unsigned int, std::vector<unsigned int>> ref_points_map;
  for (unsigned int r = 0; r < gmsh_data.__points.size(); ++r) {
    const unsigned int point_number = gmsh_data.__points[r];
    const unsigned int& ref         = gmsh_data.__points_ref[r];
    ref_points_map[ref].push_back(point_number);
  }

  for (const auto& ref_point_list : ref_points_map) {
    Array<NodeId> point_list(ref_point_list.second.size());
    for (size_t j = 0; j < ref_point_list.second.size(); ++j) {
      point_list[j] = ref_point_list.second[j];
    }
    const GmshReader::PhysicalRefId& physical_ref_id = gmsh_data.m_physical_ref_map.at(ref_point_list.first);
    descriptor.addRefItemList(RefNodeList(physical_ref_id.refId(), point_list));
  }

  descriptor.cell_owner_vector.resize(nb_cells);
  std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

  descriptor.face_owner_vector.resize(descriptor.face_number_vector.size());
  std::fill(descriptor.face_owner_vector.begin(), descriptor.face_owner_vector.end(), parallel::rank());

  descriptor.edge_owner_vector.resize(descriptor.edge_number_vector.size());
  std::fill(descriptor.edge_owner_vector.begin(), descriptor.edge_owner_vector.end(), parallel::rank());

  descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
  std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

  m_connectivity = Connectivity3D::build(descriptor);
}

GmshReader::GmshReader(const std::string& filename) : m_filename(filename)
{
  if (parallel::rank() == 0) {
    m_fin.open(m_filename);
    if (not m_fin) {
      std::ostringstream os;
      os << "cannot read file '" << m_filename << "'";
      throw NormalError(os.str());
    }

    // gmsh 2.2 format keywords
    __keywordList["$MeshFormat"]       = MESHFORMAT;
    __keywordList["$EndMeshFormat"]    = ENDMESHFORMAT;
    __keywordList["$Nodes"]            = NODES;
    __keywordList["$EndNodes"]         = ENDNODES;
    __keywordList["$Elements"]         = ELEMENTS;
    __keywordList["$EndElements"]      = ENDELEMENTS;
    __keywordList["$PhysicalNames"]    = PHYSICALNAMES;
    __keywordList["$EndPhysicalNames"] = ENDPHYSICALNAMES;
    __keywordList["$Periodic"]         = PERIODIC;
    __keywordList["$EndPeriodic"]      = ENDPERIODIC;

    __numberOfPrimitiveNodes.resize(16);
    __numberOfPrimitiveNodes[0]  = 2;    // edge
    __numberOfPrimitiveNodes[1]  = 3;    // triangle
    __numberOfPrimitiveNodes[2]  = 4;    // quadrangle
    __numberOfPrimitiveNodes[3]  = 4;    // Tetrahedron
    __numberOfPrimitiveNodes[4]  = 8;    // Hexaredron
    __numberOfPrimitiveNodes[5]  = 6;    // Prism
    __numberOfPrimitiveNodes[6]  = 5;    // Pyramid
    __numberOfPrimitiveNodes[7]  = 3;    // second order edge
    __numberOfPrimitiveNodes[8]  = 6;    // second order triangle
    __numberOfPrimitiveNodes[9]  = 9;    // second order quadrangle
    __numberOfPrimitiveNodes[10] = 10;   // second order tetrahedron
    __numberOfPrimitiveNodes[11] = 27;   // second order hexahedron
    __numberOfPrimitiveNodes[12] = 18;   // second order prism
    __numberOfPrimitiveNodes[13] = 14;   // second order pyramid
    __numberOfPrimitiveNodes[14] = 1;    // point

    __primitivesNames[0]      = "edges";
    __supportedPrimitives[0]  = true;
    __primitivesNames[1]      = "triangles";
    __supportedPrimitives[1]  = true;
    __primitivesNames[2]      = "quadrangles";
    __supportedPrimitives[2]  = true;
    __primitivesNames[3]      = "tetrahedra";
    __supportedPrimitives[3]  = true;
    __primitivesNames[4]      = "hexahedra";
    __supportedPrimitives[4]  = true;
    __primitivesNames[5]      = "prisms";
    __supportedPrimitives[5]  = true;
    __primitivesNames[6]      = "pyramids";
    __supportedPrimitives[6]  = true;
    __primitivesNames[7]      = "second order edges";
    __supportedPrimitives[7]  = false;
    __primitivesNames[8]      = "second order triangles";
    __supportedPrimitives[8]  = false;
    __primitivesNames[9]      = "second order quadrangles";
    __supportedPrimitives[9]  = false;
    __primitivesNames[10]     = "second order tetrahedra";
    __supportedPrimitives[10] = false;
    __primitivesNames[11]     = "second order hexahedra";
    __supportedPrimitives[11] = false;
    __primitivesNames[12]     = "second order prisms";
    __supportedPrimitives[12] = false;
    __primitivesNames[13]     = "second order pyramids";
    __supportedPrimitives[13] = false;
    __primitivesNames[14]     = "point";
    __supportedPrimitives[14] = true;

    std::cout << "Reading file '" << m_filename << "'\n";

    // Getting vertices list
    GmshReader::Keyword kw = this->__nextKeyword();
    switch (kw.second) {
    // case NOD: {
    //   this->__readGmsh1();
    //   break;
    // }
    case MESHFORMAT: {
      double fileVersion = this->_getReal();
      if (fileVersion != 2.2) {
        throw NormalError("Cannot read Gmsh format '" + std::to_string(fileVersion) + "'");
      }
      int fileType = this->_getInteger();
      if ((fileType < 0) or (fileType > 1)) {
        throw NormalError("Cannot read Gmsh file type '" + std::to_string(fileType) + "'");
      }

      int dataSize = this->_getInteger();
      if (dataSize != sizeof(double)) {
        throw NormalError("Data size not supported '" + std::to_string(dataSize) + "'");
      }

      kw = this->__nextKeyword();
      if (kw.second != ENDMESHFORMAT) {
        throw NormalError("reading file '" + m_filename + "': expecting $EndMeshFormat, '" + kw.first + "' was found");
      }

      this->__readGmshFormat2_2();

      break;
    }
    default: {
      throw NormalError("cannot determine format version of '" + m_filename + "'");
    }
    }

    this->__proceedData();
  }
  std::cout << std::flush;
  if (parallel::size() > 1) {
    std::cout << "Sequential mesh read! Need to be dispatched\n" << std::flush;

    const int mesh_dimension = [&]() {
      int mesh_dimension = -1;   // unknown mesh dimension
      if (m_mesh) {
        mesh_dimension = m_mesh->dimension();
      }

      Array<int> dimensions = parallel::allGather(mesh_dimension);
      std::set<int> dimension_set;
      for (size_t i = 0; i < dimensions.size(); ++i) {
        const int i_dimension = dimensions[i];
        if (i_dimension != -1) {
          dimension_set.insert(i_dimension);
        }
      }
      if (dimension_set.size() != 1) {
        throw NormalError("dimensions of read mesh parts differ!\n");
      }

      return *begin(dimension_set);
    }();
    switch (mesh_dimension) {
    case 1: {
      this->_dispatch<1>();
      break;
    }
    case 2: {
      this->_dispatch<2>();
      break;
    }
    case 3: {
      this->_dispatch<3>();
      break;
    }
    default: {
      throw NormalError("invalid mesh dimension" + std::to_string((mesh_dimension)));
    }
    }
  }
}

void
GmshReader::__readVertices()
{
  const int numberOfVerices = this->_getInteger();
  std::cout << "- Number Of Vertices: " << numberOfVerices << '\n';
  if (numberOfVerices < 0) {
    throw NormalError("reading file '" + this->m_filename + "': number of vertices is negative");
  }

  m_mesh_data.__verticesNumbers.resize(numberOfVerices);
  m_mesh_data.__vertices = Array<R3>(numberOfVerices);

  for (int i = 0; i < numberOfVerices; ++i) {
    m_mesh_data.__verticesNumbers[i] = this->_getInteger();
    const double x                   = this->_getReal();
    const double y                   = this->_getReal();
    const double z                   = this->_getReal();
    m_mesh_data.__vertices[i]        = TinyVector<3, double>(x, y, z);
  }
}

void
GmshReader::__readElements2_2()
{
  const int numberOfElements = this->_getInteger();
  std::cout << "- Number Of Elements: " << numberOfElements << '\n';
  if (numberOfElements < 0) {
    throw NormalError("reading file '" + m_filename + "': number of elements is negative");
  }

  m_mesh_data.__elementNumber.resize(numberOfElements);
  m_mesh_data.__elementType.resize(numberOfElements);
  m_mesh_data.__references.resize(numberOfElements);
  m_mesh_data.__elementVertices.resize(numberOfElements);

  for (int i = 0; i < numberOfElements; ++i) {
    m_mesh_data.__elementNumber[i] = this->_getInteger();
    const int elementType          = this->_getInteger() - 1;

    if ((elementType < 0) or (elementType > 14)) {
      throw NormalError("reading file '" + m_filename + "': unknown element type '" + std::to_string(elementType) +
                        "'");
    }
    m_mesh_data.__elementType[i] = elementType;
    const int numberOfTags       = this->_getInteger();
    m_mesh_data.__references[i]  = this->_getInteger();   // physical reference
    for (int tag = 1; tag < numberOfTags; ++tag) {
      this->_getInteger();   // drops remaining tags
    }

    const int numberOfVertices = __numberOfPrimitiveNodes[elementType];
    m_mesh_data.__elementVertices[i].resize(numberOfVertices);
    for (int j = 0; j < numberOfVertices; ++j) {
      m_mesh_data.__elementVertices[i][j] = this->_getInteger();
    }
  }
}

void
GmshReader::__readPhysicalNames2_2()
{
  const int number_of_names = this->_getInteger();
  for (int i = 0; i < number_of_names; ++i) {
    const int physical_dimension = this->_getInteger();
    const int physical_number    = this->_getInteger();
    std::string physical_name;
    m_fin >> physical_name;
    physical_name = std::regex_replace(physical_name, std::regex("(\")"), "");

    PhysicalRefId physical_ref_id(physical_dimension, RefId(physical_number, physical_name));

    if (auto i_searched_physical_ref_id = m_mesh_data.m_physical_ref_map.find(physical_number);
        i_searched_physical_ref_id != m_mesh_data.m_physical_ref_map.end()) {
      std::stringstream os;
      os << "Physical reference id '" << physical_ref_id << "' already defined as '"
         << i_searched_physical_ref_id->second << "'!";
      throw NormalError(os.str());
    }
    m_mesh_data.m_physical_ref_map[physical_number] = physical_ref_id;
  }
}

void
GmshReader::__readPeriodic2_2()
{
  // This is just a compatibility reading, periodic information is not
  // used
  const int number_of_periodic = this->_getInteger();
  for (int i = 0; i < number_of_periodic; ++i) {
    [[maybe_unused]] const int dimension              = this->_getInteger();
    [[maybe_unused]] const int id                     = this->_getInteger();
    [[maybe_unused]] const int master_id              = this->_getInteger();
    [[maybe_unused]] const int nb_corresponding_nodes = this->_getInteger();
    for (int i_node = 0; i_node < nb_corresponding_nodes; ++i_node) {
      [[maybe_unused]] const int node_id   = this->_getInteger();
      [[maybe_unused]] const int master_id = this->_getInteger();
    }
  }
}

// std::shared_ptr<IConnectivity>
// GmshReader::_buildConnectivity3D(const size_t nb_cells)
// {
//   ConnectivityDescriptor descriptor;

//   descriptor.node_number_vector = m_mesh_data.__verticesNumbers;
//   descriptor.cell_type_vector.resize(nb_cells);
//   descriptor.cell_number_vector.resize(nb_cells);
//   descriptor.cell_to_node_vector.resize(nb_cells);

//   const size_t nb_tetrahedra = m_mesh_data.__tetrahedra.size();
//   for (size_t j = 0; j < nb_tetrahedra; ++j) {
//     descriptor.cell_to_node_vector[j].resize(4);
//     for (int r = 0; r < 4; ++r) {
//       descriptor.cell_to_node_vector[j][r] = m_mesh_data.__tetrahedra[j][r];
//     }
//     descriptor.cell_type_vector[j]   = CellType::Tetrahedron;
//     descriptor.cell_number_vector[j] = m_mesh_data.__tetrahedra_number[j];
//   }
//   const size_t nb_hexahedra = m_mesh_data.__hexahedra.size();
//   for (size_t j = 0; j < nb_hexahedra; ++j) {
//     const size_t jh = nb_tetrahedra + j;
//     descriptor.cell_to_node_vector[jh].resize(8);
//     for (int r = 0; r < 8; ++r) {
//       descriptor.cell_to_node_vector[jh][r] = m_mesh_data.__hexahedra[j][r];
//     }
//     descriptor.cell_type_vector[jh]   = CellType::Hexahedron;
//     descriptor.cell_number_vector[jh] = m_mesh_data.__hexahedra_number[j];
//   }

//   std::map<unsigned int, std::vector<unsigned int>> ref_cells_map;
//   for (unsigned int r = 0; r < m_mesh_data.__tetrahedra_ref.size(); ++r) {
//     const unsigned int elem_number = m_mesh_data.__tetrahedra_ref[r];
//     const unsigned int& ref        = m_mesh_data.__tetrahedra_ref[r];
//     ref_cells_map[ref].push_back(elem_number);
//   }

//   for (unsigned int j = 0; j < m_mesh_data.__hexahedra_ref.size(); ++j) {
//     const size_t elem_number = nb_tetrahedra + j;
//     const unsigned int& ref  = m_mesh_data.__hexahedra_ref[j];
//     ref_cells_map[ref].push_back(elem_number);
//   }

//   for (const auto& ref_cell_list : ref_cells_map) {
//     Array<CellId> cell_list(ref_cell_list.second.size());
//     for (size_t j = 0; j < ref_cell_list.second.size(); ++j) {
//       cell_list[j] = ref_cell_list.second[j];
//     }
//     const PhysicalRefId& physical_ref_id = m_mesh_data.m_physical_ref_map.at(ref_cell_list.first);
//     descriptor.addRefItemList(RefCellList(physical_ref_id.refId(), cell_list));
//   }

//   ConnectivityBuilderBase::_computeCellFaceAndFaceNodeConnectivities<3>(descriptor);

//   const auto& node_number_vector = descriptor.node_number_vector;

//   {
//     using Face                                                                 = ConnectivityFace<3>;
//     const std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map = [&] {
//       std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map;
//       for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
//         const auto& node_vector                               = descriptor.face_to_node_vector[l];
//         face_to_id_map[Face(node_vector, node_number_vector)] = l;
//       }
//       return face_to_id_map;
//     }();

//     std::unordered_map<int, FaceId> face_number_id_map = [&] {
//       std::unordered_map<int, FaceId> face_number_id_map;
//       for (size_t l = 0; l < descriptor.face_number_vector.size(); ++l) {
//         face_number_id_map[descriptor.face_number_vector[l]] = l;
//       }
//       Assert(face_number_id_map.size() == descriptor.face_number_vector.size());
//       return face_number_id_map;
//     }();

//     std::map<unsigned int, std::vector<unsigned int>> ref_faces_map;
//     for (unsigned int f = 0; f < m_mesh_data.__triangles.size(); ++f) {
//       const unsigned int face_id = [&] {
//         auto i = face_to_id_map.find(
//           Face({m_mesh_data.__triangles[f][0], m_mesh_data.__triangles[f][1], m_mesh_data.__triangles[f][2]},
//                node_number_vector));
//         if (i == face_to_id_map.end()) {
//           throw NormalError("face not found");
//         }
//         return i->second;
//       }();

//       const unsigned int& ref = m_mesh_data.__triangles_ref[f];
//       ref_faces_map[ref].push_back(face_id);

//       if (descriptor.face_number_vector[face_id] != m_mesh_data.__quadrangles_number[f]) {
//         if (auto i_face = face_number_id_map.find(m_mesh_data.__quadrangles_number[f]);
//             i_face != face_number_id_map.end()) {
//           const int other_face_id = i_face->second;
//           std::swap(descriptor.face_number_vector[face_id], descriptor.face_number_vector[other_face_id]);

//           face_number_id_map.erase(descriptor.face_number_vector[face_id]);
//           face_number_id_map.erase(descriptor.face_number_vector[other_face_id]);

//           face_number_id_map[descriptor.face_number_vector[face_id]]       = face_id;
//           face_number_id_map[descriptor.face_number_vector[other_face_id]] = other_face_id;
//         } else {
//           face_number_id_map.erase(descriptor.face_number_vector[face_id]);
//           descriptor.face_number_vector[face_id]                     = m_mesh_data.__quadrangles_number[f];
//           face_number_id_map[descriptor.face_number_vector[face_id]] = face_id;
//         }
//       }
//     }

//     for (unsigned int f = 0; f < m_mesh_data.__quadrangles.size(); ++f) {
//       const unsigned int face_id = [&] {
//         auto i = face_to_id_map.find(Face({m_mesh_data.__quadrangles[f][0], m_mesh_data.__quadrangles[f][1],
//                                            m_mesh_data.__quadrangles[f][2], m_mesh_data.__quadrangles[f][3]},
//                                           node_number_vector));
//         if (i == face_to_id_map.end()) {
//           throw NormalError("face not found");
//         }
//         return i->second;
//       }();

//       const unsigned int& ref = m_mesh_data.__quadrangles_ref[f];
//       ref_faces_map[ref].push_back(face_id);

//       if (descriptor.face_number_vector[face_id] != m_mesh_data.__quadrangles_number[f]) {
//         if (auto i_face = face_number_id_map.find(m_mesh_data.__quadrangles_number[f]);
//             i_face != face_number_id_map.end()) {
//           const int other_face_id = i_face->second;
//           std::swap(descriptor.face_number_vector[face_id], descriptor.face_number_vector[other_face_id]);

//           face_number_id_map.erase(descriptor.face_number_vector[face_id]);
//           face_number_id_map.erase(descriptor.face_number_vector[other_face_id]);

//           face_number_id_map[descriptor.face_number_vector[face_id]]       = face_id;
//           face_number_id_map[descriptor.face_number_vector[other_face_id]] = other_face_id;
//         } else {
//           face_number_id_map.erase(descriptor.face_number_vector[face_id]);
//           descriptor.face_number_vector[face_id]                     = m_mesh_data.__quadrangles_number[f];
//           face_number_id_map[descriptor.face_number_vector[face_id]] = face_id;
//         }
//       }
//     }

//     for (const auto& ref_face_list : ref_faces_map) {
//       Array<FaceId> face_list(ref_face_list.second.size());
//       for (size_t j = 0; j < ref_face_list.second.size(); ++j) {
//         face_list[j] = ref_face_list.second[j];
//       }
//       const PhysicalRefId& physical_ref_id = m_mesh_data.m_physical_ref_map.at(ref_face_list.first);
//       descriptor.addRefItemList(RefFaceList{physical_ref_id.refId(), face_list});
//     }
//   }

//   ConnectivityBuilderBase::_computeFaceEdgeAndEdgeNodeAndCellEdgeConnectivities<3>(descriptor);

//   {
//     using Edge                                                                 = ConnectivityFace<2>;
//     const auto& node_number_vector                                             = descriptor.node_number_vector;
//     const std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map = [&] {
//       std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map;
//       for (EdgeId l = 0; l < descriptor.edge_to_node_vector.size(); ++l) {
//         const auto& node_vector                               = descriptor.edge_to_node_vector[l];
//         edge_to_id_map[Edge(node_vector, node_number_vector)] = l;
//       }
//       return edge_to_id_map;
//     }();

//     std::unordered_map<int, EdgeId> edge_number_id_map = [&] {
//       std::unordered_map<int, EdgeId> edge_number_id_map;
//       for (size_t l = 0; l < descriptor.edge_number_vector.size(); ++l) {
//         edge_number_id_map[descriptor.edge_number_vector[l]] = l;
//       }
//       Assert(edge_number_id_map.size() == descriptor.edge_number_vector.size());
//       return edge_number_id_map;
//     }();

//     std::map<unsigned int, std::vector<unsigned int>> ref_edges_map;
//     for (unsigned int e = 0; e < m_mesh_data.__edges.size(); ++e) {
//       const unsigned int edge_id = [&] {
//         auto i = edge_to_id_map.find(Edge({m_mesh_data.__edges[e][0], m_mesh_data.__edges[e][1]},
//         node_number_vector)); if (i == edge_to_id_map.end()) {
//           std::stringstream error_msg;
//           error_msg << "edge " << m_mesh_data.__edges[e][0] << " not found";
//           throw NormalError(error_msg.str());
//         }
//         return i->second;
//       }();
//       const unsigned int& ref = m_mesh_data.__edges_ref[e];
//       ref_edges_map[ref].push_back(edge_id);

//       if (descriptor.edge_number_vector[edge_id] != m_mesh_data.__edges_number[e]) {
//         if (auto i_edge = edge_number_id_map.find(m_mesh_data.__edges_number[e]); i_edge != edge_number_id_map.end())
//         {
//           const int other_edge_id = i_edge->second;
//           std::swap(descriptor.edge_number_vector[edge_id], descriptor.edge_number_vector[other_edge_id]);

//           edge_number_id_map.erase(descriptor.edge_number_vector[edge_id]);
//           edge_number_id_map.erase(descriptor.edge_number_vector[other_edge_id]);

//           edge_number_id_map[descriptor.edge_number_vector[edge_id]]       = edge_id;
//           edge_number_id_map[descriptor.edge_number_vector[other_edge_id]] = other_edge_id;
//         } else {
//           edge_number_id_map.erase(descriptor.edge_number_vector[edge_id]);
//           descriptor.edge_number_vector[edge_id]                     = m_mesh_data.__edges_number[e];
//           edge_number_id_map[descriptor.edge_number_vector[edge_id]] = edge_id;
//         }
//       }
//     }

//     for (const auto& ref_edge_list : ref_edges_map) {
//       Array<EdgeId> edge_list(ref_edge_list.second.size());
//       for (size_t j = 0; j < ref_edge_list.second.size(); ++j) {
//         edge_list[j] = ref_edge_list.second[j];
//       }
//       const PhysicalRefId& physical_ref_id = m_mesh_data.m_physical_ref_map.at(ref_edge_list.first);
//       descriptor.addRefItemList(RefEdgeList{physical_ref_id.refId(), edge_list});
//     }
//   }

//   std::map<unsigned int, std::vector<unsigned int>> ref_points_map;
//   for (unsigned int r = 0; r < m_mesh_data.__points.size(); ++r) {
//     const unsigned int point_number = m_mesh_data.__points[r];
//     const unsigned int& ref         = m_mesh_data.__points_ref[r];
//     ref_points_map[ref].push_back(point_number);
//   }

//   for (const auto& ref_point_list : ref_points_map) {
//     Array<NodeId> point_list(ref_point_list.second.size());
//     for (size_t j = 0; j < ref_point_list.second.size(); ++j) {
//       point_list[j] = ref_point_list.second[j];
//     }
//     const PhysicalRefId& physical_ref_id = m_mesh_data.m_physical_ref_map.at(ref_point_list.first);
//     descriptor.addRefItemList(RefNodeList(physical_ref_id.refId(), point_list));
//   }

//   descriptor.cell_owner_vector.resize(nb_cells);
//   std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

//   descriptor.face_owner_vector.resize(descriptor.face_number_vector.size());
//   std::fill(descriptor.face_owner_vector.begin(), descriptor.face_owner_vector.end(), parallel::rank());

//   descriptor.edge_owner_vector.resize(descriptor.edge_number_vector.size());
//   std::fill(descriptor.edge_owner_vector.begin(), descriptor.edge_owner_vector.end(), parallel::rank());

//   descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
//   std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

//   return Connectivity3D::build(descriptor);
// }

void
GmshReader::__proceedData()
{
  TinyVector<15, int> elementNumber = zero;
  std::vector<std::set<size_t>> elementReferences(15);

  for (size_t i = 0; i < m_mesh_data.__elementType.size(); ++i) {
    const int elementType = m_mesh_data.__elementType[i];
    elementNumber[elementType]++;
    elementReferences[elementType].insert(m_mesh_data.__references[i]);
  }

  for (size_t i = 0; i < elementNumber.dimension(); ++i) {
    if (elementNumber[i] > 0) {
      std::cout << "  - Number of " << __primitivesNames[i] << ": " << elementNumber[i];
      if (not(__supportedPrimitives[i])) {
        std::cout << " [" << rang::fg::yellow << "IGNORED" << rang::style::reset << "]";
      } else {
        std::cout << " references: ";
        for (std::set<size_t>::const_iterator iref = elementReferences[i].begin(); iref != elementReferences[i].end();
             ++iref) {
          if (iref != elementReferences[i].begin())
            std::cout << ',';
          std::cout << *iref;
        }

        switch (i) {
        // Supported entities
        case 0: {   // edges
          m_mesh_data.__edges = Array<GmshData::Edge>(elementNumber[i]);
          m_mesh_data.__edges_ref.resize(elementNumber[i]);
          m_mesh_data.__edges_number.resize(elementNumber[i]);
          break;
        }
        case 1: {   // triangles
          m_mesh_data.__triangles = Array<GmshData::Triangle>(elementNumber[i]);
          m_mesh_data.__triangles_ref.resize(elementNumber[i]);
          m_mesh_data.__triangles_number.resize(elementNumber[i]);
          break;
        }
        case 2: {   // quadrangles
          m_mesh_data.__quadrangles = Array<GmshData::Quadrangle>(elementNumber[i]);
          m_mesh_data.__quadrangles_ref.resize(elementNumber[i]);
          m_mesh_data.__quadrangles_number.resize(elementNumber[i]);
          break;
        }
        case 3: {   // tetrahedra
          m_mesh_data.__tetrahedra = Array<GmshData::Tetrahedron>(elementNumber[i]);
          m_mesh_data.__tetrahedra_ref.resize(elementNumber[i]);
          m_mesh_data.__tetrahedra_number.resize(elementNumber[i]);
          break;
        }
        case 4: {   // hexahedra
          m_mesh_data.__hexahedra = Array<GmshData::Hexahedron>(elementNumber[i]);
          m_mesh_data.__hexahedra_ref.resize(elementNumber[i]);
          m_mesh_data.__hexahedra_number.resize(elementNumber[i]);
          break;
        }
        case 5: {   // prism
          m_mesh_data.__prisms = Array<GmshData::Prism>(elementNumber[i]);
          m_mesh_data.__prisms_ref.resize(elementNumber[i]);
          m_mesh_data.__prisms_number.resize(elementNumber[i]);
          break;
        }
        case 6: {   // pyramid
          m_mesh_data.__pyramids = Array<GmshData::Pyramid>(elementNumber[i]);
          m_mesh_data.__pyramids_ref.resize(elementNumber[i]);
          m_mesh_data.__pyramids_number.resize(elementNumber[i]);
          break;
        }
        case 14: {   // point
          m_mesh_data.__points = Array<GmshData::Point>(elementNumber[i]);
          m_mesh_data.__points_ref.resize(elementNumber[i]);
          m_mesh_data.__points_number.resize(elementNumber[i]);
          break;
        }
          // Unsupported entities
        case 7:    // second order edge
        case 8:    // second order triangle
        case 9:    // second order quadrangle
        case 10:   // second order tetrahedron
        case 11:   // second order hexahedron
        case 12:   // second order prism
        case 13:   // second order pyramid
        default: {
          throw NormalError("reading file '" + m_filename + "': ff3d cannot read this kind of element");
        }
        }
      }
      std::cout << '\n';
    }
  }

  std::cout << "- Building correspondance table\n";
  int minNumber = std::numeric_limits<int>::max();
  int maxNumber = std::numeric_limits<int>::min();
  for (size_t i = 0; i < m_mesh_data.__verticesNumbers.size(); ++i) {
    const int& vertexNumber = m_mesh_data.__verticesNumbers[i];
    minNumber               = std::min(minNumber, vertexNumber);
    maxNumber               = std::max(maxNumber, vertexNumber);
  }

  if (minNumber < 0) {
    throw NotImplementedError("reading file '" + m_filename + "': negative vertices number");
  }

  // A value of -1 means that the vertex is unknown
  m_mesh_data.__verticesCorrepondance.resize(maxNumber + 1, -1);

  for (size_t i = 0; i < m_mesh_data.__verticesNumbers.size(); ++i) {
    m_mesh_data.__verticesCorrepondance[m_mesh_data.__verticesNumbers[i]] = i;
  }

  // reset element number to count them while filling structures
  elementNumber = zero;

  // Element structures filling
  for (size_t i = 0; i < m_mesh_data.__elementType.size(); ++i) {
    std::vector<int>& elementVertices = m_mesh_data.__elementVertices[i];
    switch (m_mesh_data.__elementType[i]) {
    // Supported entities
    case 0: {   // edge
      int& edgeNumber = elementNumber[0];
      const int a     = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      const int b     = m_mesh_data.__verticesCorrepondance[elementVertices[1]];
      if ((a < 0) or (b < 0)) {
        throw NormalError("reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                          " [bad vertices definition]");
      }
      m_mesh_data.__edges[edgeNumber]        = GmshData::Edge(a, b);
      m_mesh_data.__edges_ref[edgeNumber]    = m_mesh_data.__references[i];
      m_mesh_data.__edges_number[edgeNumber] = m_mesh_data.__elementNumber[i];
      edgeNumber++;   // one more edge
      break;
    }
    case 1: {   // triangles
      int& triangleNumber = elementNumber[1];

      const int a = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      const int b = m_mesh_data.__verticesCorrepondance[elementVertices[1]];
      const int c = m_mesh_data.__verticesCorrepondance[elementVertices[2]];
      if ((a < 0) or (b < 0) or (c < 0)) {
        throw NormalError("reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                          " [bad vertices definition]");
      }
      m_mesh_data.__triangles[triangleNumber]        = GmshData::Triangle(a, b, c);
      m_mesh_data.__triangles_ref[triangleNumber]    = m_mesh_data.__references[i];
      m_mesh_data.__triangles_number[triangleNumber] = m_mesh_data.__elementNumber[i];
      triangleNumber++;   // one more triangle
      break;
    }
    case 2: {   // quadrangle
      int& quadrilateralNumber = elementNumber[2];

      const int a = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      const int b = m_mesh_data.__verticesCorrepondance[elementVertices[1]];
      const int c = m_mesh_data.__verticesCorrepondance[elementVertices[2]];
      const int d = m_mesh_data.__verticesCorrepondance[elementVertices[3]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0)) {
        throw NormalError("reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                          " [bad vertices definition]");
      }
      m_mesh_data.__quadrangles[quadrilateralNumber]        = GmshData::Quadrangle(a, b, c, d);
      m_mesh_data.__quadrangles_ref[quadrilateralNumber]    = m_mesh_data.__references[i];
      m_mesh_data.__quadrangles_number[quadrilateralNumber] = m_mesh_data.__elementNumber[i];
      quadrilateralNumber++;   // one more quadrangle
      break;
    }
    case 3: {   // tetrahedra
      int& tetrahedronNumber = elementNumber[3];

      const int a = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      const int b = m_mesh_data.__verticesCorrepondance[elementVertices[1]];
      const int c = m_mesh_data.__verticesCorrepondance[elementVertices[2]];
      const int d = m_mesh_data.__verticesCorrepondance[elementVertices[3]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0)) {
        throw NormalError("reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                          " [bad vertices definition]");
      }
      m_mesh_data.__tetrahedra[tetrahedronNumber]        = GmshData::Tetrahedron(a, b, c, d);
      m_mesh_data.__tetrahedra_ref[tetrahedronNumber]    = m_mesh_data.__references[i];
      m_mesh_data.__tetrahedra_number[tetrahedronNumber] = m_mesh_data.__elementNumber[i];
      tetrahedronNumber++;   // one more tetrahedron
      break;
    }
    case 4: {   // hexaredron
      int& hexahedronNumber = elementNumber[4];

      const int a = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      const int b = m_mesh_data.__verticesCorrepondance[elementVertices[1]];
      const int c = m_mesh_data.__verticesCorrepondance[elementVertices[2]];
      const int d = m_mesh_data.__verticesCorrepondance[elementVertices[3]];
      const int e = m_mesh_data.__verticesCorrepondance[elementVertices[4]];
      const int f = m_mesh_data.__verticesCorrepondance[elementVertices[5]];
      const int g = m_mesh_data.__verticesCorrepondance[elementVertices[6]];
      const int h = m_mesh_data.__verticesCorrepondance[elementVertices[7]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0) or (e < 0) or (f < 0) or (g < 0) or (h < 0)) {
        throw NormalError("reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                          " [bad vertices definition]");
      }
      m_mesh_data.__hexahedra[hexahedronNumber]        = GmshData::Hexahedron(a, b, c, d, e, f, g, h);
      m_mesh_data.__hexahedra_ref[hexahedronNumber]    = m_mesh_data.__references[i];
      m_mesh_data.__hexahedra_number[hexahedronNumber] = m_mesh_data.__elementNumber[i];
      hexahedronNumber++;   // one more hexahedron
      break;
    }
    case 5: {   // prism
      int& prism_number = elementNumber[5];
      const int a       = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      const int b       = m_mesh_data.__verticesCorrepondance[elementVertices[1]];
      const int c       = m_mesh_data.__verticesCorrepondance[elementVertices[2]];
      const int d       = m_mesh_data.__verticesCorrepondance[elementVertices[3]];
      const int e       = m_mesh_data.__verticesCorrepondance[elementVertices[4]];
      const int f       = m_mesh_data.__verticesCorrepondance[elementVertices[5]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0) or (e < 0) or (f < 0)) {
        throw NormalError("reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                          " [bad vertices definition]");
      }
      m_mesh_data.__prisms[prism_number]        = GmshData::Prism(a, b, c, d, e, f);
      m_mesh_data.__prisms_ref[prism_number]    = m_mesh_data.__references[i];
      m_mesh_data.__prisms_number[prism_number] = m_mesh_data.__elementNumber[i];
      prism_number++;   // one more prism
      break;
    }
    case 6: {   // pyramid
      int& pyramid_number = elementNumber[6];

      const int a = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      const int b = m_mesh_data.__verticesCorrepondance[elementVertices[1]];
      const int c = m_mesh_data.__verticesCorrepondance[elementVertices[2]];
      const int d = m_mesh_data.__verticesCorrepondance[elementVertices[3]];
      const int e = m_mesh_data.__verticesCorrepondance[elementVertices[4]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0) or (e < 0)) {
        throw NormalError("reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                          " [bad vertices definition]");
      }
      m_mesh_data.__pyramids[pyramid_number]        = GmshData::Pyramid(a, b, c, d, e);
      m_mesh_data.__pyramids_ref[pyramid_number]    = m_mesh_data.__references[i];
      m_mesh_data.__pyramids_number[pyramid_number] = m_mesh_data.__elementNumber[i];
      pyramid_number++;   // one more hexahedron
      break;
    }
      // Unsupported entities
    case 14: {   // point
      int& point_number                         = elementNumber[14];
      const int a                               = m_mesh_data.__verticesCorrepondance[elementVertices[0]];
      m_mesh_data.__points[point_number]        = a;
      m_mesh_data.__points_ref[point_number]    = m_mesh_data.__references[i];
      m_mesh_data.__points_number[point_number] = m_mesh_data.__elementNumber[i];
      point_number++;
      break;
    }
    case 7:      // second order edge
    case 8:      // second order triangle
    case 9:      // second order quadrangle
    case 10:     // second order tetrahedron
    case 11:     // second order hexahedron
    case 12:     // second order prism
    case 13: {   // second order pyramid
    }
    default: {
      throw NormalError("reading file '" + m_filename + "': ff3d cannot read this kind of element");
    }
    }
  }

  TinyVector<15, int> dimension0_mask = zero;
  dimension0_mask[14]                 = 1;
  TinyVector<15, int> dimension1_mask = zero;
  dimension1_mask[0]                  = 1;
  dimension1_mask[7]                  = 1;
  TinyVector<15, int> dimension2_mask = zero;
  dimension2_mask[1]                  = 1;
  dimension2_mask[2]                  = 1;
  dimension2_mask[8]                  = 1;
  dimension2_mask[9]                  = 1;
  TinyVector<15, int> dimension3_mask = zero;
  dimension3_mask[3]                  = 1;
  dimension3_mask[4]                  = 1;
  dimension3_mask[5]                  = 1;
  dimension3_mask[6]                  = 1;
  dimension3_mask[10]                 = 1;
  dimension3_mask[11]                 = 1;
  dimension3_mask[12]                 = 1;
  dimension3_mask[13]                 = 1;

  std::cout << "- dimension 0 entities: " << dot(dimension0_mask, elementNumber) << '\n';
  std::cout << "- dimension 1 entities: " << dot(dimension1_mask, elementNumber) << '\n';
  std::cout << "- dimension 2 entities: " << dot(dimension2_mask, elementNumber) << '\n';
  std::cout << "- dimension 3 entities: " << dot(dimension3_mask, elementNumber) << '\n';
  if (dot(dimension3_mask, elementNumber) > 0) {
    const size_t nb_cells = dot(dimension3_mask, elementNumber);

    GmshConnectivityBuilder<3> connectivity_builder(m_mesh_data, nb_cells);

    std::shared_ptr p_connectivity =
      std::dynamic_pointer_cast<const Connectivity3D>(connectivity_builder.connectivity());
    const Connectivity3D& connectivity = *p_connectivity;

    using MeshType = Mesh<Connectivity3D>;
    using Rd       = TinyVector<3, double>;

    NodeValue<Rd> xr(connectivity);
    for (NodeId i = 0; i < m_mesh_data.__vertices.size(); ++i) {
      xr[i][0] = m_mesh_data.__vertices[i][0];
      xr[i][1] = m_mesh_data.__vertices[i][1];
      xr[i][2] = m_mesh_data.__vertices[i][2];
    }
    m_mesh = std::make_shared<MeshType>(p_connectivity, xr);

  } else if (dot(dimension2_mask, elementNumber) > 0) {
    const size_t nb_cells = dot(dimension2_mask, elementNumber);

    GmshConnectivityBuilder<2> connectivity_builder(m_mesh_data, nb_cells);

    std::shared_ptr p_connectivity =
      std::dynamic_pointer_cast<const Connectivity2D>(connectivity_builder.connectivity());
    const Connectivity2D& connectivity = *p_connectivity;

    using MeshType = Mesh<Connectivity2D>;
    using Rd       = TinyVector<2, double>;

    NodeValue<Rd> xr(connectivity);
    for (NodeId i = 0; i < m_mesh_data.__vertices.size(); ++i) {
      xr[i][0] = m_mesh_data.__vertices[i][0];
      xr[i][1] = m_mesh_data.__vertices[i][1];
    }
    m_mesh = std::make_shared<MeshType>(p_connectivity, xr);

  } else if (dot(dimension1_mask, elementNumber) > 0) {
    const size_t nb_cells = dot(dimension1_mask, elementNumber);

    GmshConnectivityBuilder<1> connectivity_builder(m_mesh_data, nb_cells);

    std::shared_ptr p_connectivity =
      std::dynamic_pointer_cast<const Connectivity1D>(connectivity_builder.connectivity());
    const Connectivity1D& connectivity = *p_connectivity;

    using MeshType = Mesh<Connectivity1D>;
    using Rd       = TinyVector<1, double>;

    NodeValue<Rd> xr(connectivity);
    for (NodeId i = 0; i < m_mesh_data.__vertices.size(); ++i) {
      xr[i][0] = m_mesh_data.__vertices[i][0];
    }

    m_mesh = std::make_shared<MeshType>(p_connectivity, xr);

  } else {
    throw NormalError("using a dimension 0 mesh is forbidden");
  }
}

GmshReader::Keyword
GmshReader::__nextKeyword()
{
  GmshReader::Keyword kw;

  std::string aKeyword;
  m_fin >> aKeyword;
  if (not m_fin) {
    kw.second = EndOfFile;
    return kw;
  }

  KeywordList::iterator i = __keywordList.find(aKeyword.c_str());

  if (i != __keywordList.end()) {
    kw.first  = (*i).first;
    kw.second = (*i).second;
    return kw;
  }

  throw NormalError("reading file '" + m_filename + "': unknown keyword '" + aKeyword + "'");

  kw.first  = aKeyword;
  kw.second = Unknown;
  return kw;
}

void
GmshReader::__readGmshFormat2_2()
{
  std::cout << "- Reading Gmsh format 2.2\n";
  GmshReader::Keyword kw = std::make_pair("", Unknown);
  while (kw.second != EndOfFile) {
    kw = this->__nextKeyword();
    switch (kw.second) {
    case NODES: {
      this->__readVertices();
      if (this->__nextKeyword().second != ENDNODES) {
        throw NormalError("reading file '" + m_filename + "': expecting $EndNodes, '" + kw.first + "' was found");
      }
      break;
    }
    case ELEMENTS: {
      this->__readElements2_2();
      kw = this->__nextKeyword();
      if (kw.second != ENDELEMENTS) {
        throw NormalError("reading file '" + m_filename + "': expecting $EndElements, '" + kw.first + "' was found");
      }
      break;
    }
    case PHYSICALNAMES: {
      this->__readPhysicalNames2_2();
      if (this->__nextKeyword().second != ENDPHYSICALNAMES) {
        throw NormalError("reading file '" + m_filename + "': expecting $EndPhysicalNames, '" + kw.first +
                          "' was found");
      }
      break;
    }
    case PERIODIC: {
      this->__readPeriodic2_2();
      if (this->__nextKeyword().second != ENDPERIODIC) {
        throw NormalError("reading file '" + m_filename + "': expecting $EndPeriodic, '" + kw.first + "' was found");
      }
      break;
    }

    case EndOfFile: {
      break;
    }
    default: {
      throw NormalError("reading file '" + m_filename + "': unexpected '" + kw.first + "'");
    }
    }
  }
}
