#ifndef MESH_BUILDER_BASE_HPP
#define MESH_BUILDER_BASE_HPP

#include <mesh/IMesh.hpp>
#include <mesh/ItemId.hpp>
#include <utils/PugsAssert.hpp>
#include <utils/PugsMacros.hpp>

#include <memory>
#include <vector>

class ConnectivityDescriptor;

class MeshBuilderBase
{
 protected:
  template <size_t Dimension>
  class ConnectivityFace;

  std::shared_ptr<const IMesh> m_mesh;

  template <size_t Dimension>
  static void _computeCellFaceAndFaceNodeConnectivities(ConnectivityDescriptor& descriptor);

  template <size_t Dimension>
  static void _computeFaceEdgeAndEdgeNodeAndCellEdgeConnectivities(ConnectivityDescriptor& descriptor);

  template <int Dimension>
  void _dispatch();

 public:
  std::shared_ptr<const IMesh>
  mesh() const
  {
    return m_mesh;
  }

  MeshBuilderBase()  = default;
  ~MeshBuilderBase() = default;
};

template <>
class MeshBuilderBase::ConnectivityFace<2>
{
 public:
  friend struct Hash;

  struct Hash
  {
    size_t
    operator()(const ConnectivityFace& f) const
    {
      size_t hash = 0;
      hash ^= std::hash<unsigned int>()(f.m_node0_id);
      hash ^= std::hash<unsigned int>()(f.m_node1_id) >> 1;
      return hash;
    }
  };

 private:
  const std::vector<int>& m_node_number_vector;

  unsigned int m_node0_id;
  unsigned int m_node1_id;

  bool m_reversed;

 public:
  std::vector<unsigned int>
  nodeIdList() const
  {
    return {m_node0_id, m_node1_id};
  }

  bool
  reversed() const
  {
    return m_reversed;
  }

  PUGS_INLINE
  bool
  operator==(const ConnectivityFace& f) const
  {
    return ((m_node0_id == f.m_node0_id) and (m_node1_id == f.m_node1_id));
  }

  PUGS_INLINE
  bool
  operator<(const ConnectivityFace& f) const
  {
    return ((m_node_number_vector[m_node0_id] < m_node_number_vector[f.m_node0_id]) or
            ((m_node_number_vector[m_node0_id] == m_node_number_vector[f.m_node0_id]) and
             (m_node_number_vector[m_node1_id] < m_node_number_vector[f.m_node1_id])));
  }

  PUGS_INLINE
  ConnectivityFace(const std::vector<unsigned int>& node_id_list, const std::vector<int>& node_number_vector)
    : m_node_number_vector(node_number_vector)
  {
    Assert(node_id_list.size() == 2);

    if (m_node_number_vector[node_id_list[0]] < m_node_number_vector[node_id_list[1]]) {
      m_node0_id = node_id_list[0];
      m_node1_id = node_id_list[1];
      m_reversed = false;
    } else {
      m_node0_id = node_id_list[1];
      m_node1_id = node_id_list[0];
      m_reversed = true;
    }
  }

  PUGS_INLINE
  ConnectivityFace(const ConnectivityFace&) = default;

  PUGS_INLINE
  ~ConnectivityFace() = default;
};

template <>
class MeshBuilderBase::ConnectivityFace<3>
{
 public:
  friend struct Hash;

  struct Hash
  {
    size_t
    operator()(const ConnectivityFace& f) const
    {
      size_t hash = 0;
      for (size_t i = 0; i < f.m_node_id_list.size(); ++i) {
        hash ^= std::hash<unsigned int>()(f.m_node_id_list[i]) >> i;
      }
      return hash;
    }
  };

 private:
  bool m_reversed;
  std::vector<NodeId::base_type> m_node_id_list;
  const std::vector<int>& m_node_number_vector;

  PUGS_INLINE
  std::vector<unsigned int>
  _sort(const std::vector<unsigned int>& node_list)
  {
    const auto min_id = std::min_element(node_list.begin(), node_list.end());
    const int shift   = std::distance(node_list.begin(), min_id);

    std::vector<unsigned int> rotated_node_list(node_list.size());
    if (node_list[(shift + 1) % node_list.size()] > node_list[(shift + node_list.size() - 1) % node_list.size()]) {
      for (size_t i = 0; i < node_list.size(); ++i) {
        rotated_node_list[i] = node_list[(shift + node_list.size() - i) % node_list.size()];
        m_reversed           = true;
      }
    } else {
      for (size_t i = 0; i < node_list.size(); ++i) {
        rotated_node_list[i] = node_list[(shift + i) % node_list.size()];
      }
    }

    return rotated_node_list;
  }

 public:
  PUGS_INLINE
  const bool&
  reversed() const
  {
    return m_reversed;
  }

  PUGS_INLINE
  const std::vector<unsigned int>&
  nodeIdList() const
  {
    return m_node_id_list;
  }

  PUGS_INLINE
  ConnectivityFace(const std::vector<unsigned int>& given_node_id_list, const std::vector<int>& node_number_vector)
    : m_reversed(false), m_node_id_list(_sort(given_node_id_list)), m_node_number_vector(node_number_vector)
  {
    ;
  }

 public:
  bool
  operator==(const ConnectivityFace& f) const
  {
    if (m_node_id_list.size() == f.nodeIdList().size()) {
      for (size_t j = 0; j < m_node_id_list.size(); ++j) {
        if (m_node_id_list[j] != f.nodeIdList()[j]) {
          return false;
        }
      }
      return true;
    }
    return false;
  }

  PUGS_INLINE
  bool
  operator<(const ConnectivityFace& f) const
  {
    const size_t min_nb_nodes = std::min(f.m_node_id_list.size(), m_node_id_list.size());
    for (size_t i = 0; i < min_nb_nodes; ++i) {
      if (m_node_id_list[i] < f.m_node_id_list[i])
        return true;
      if (m_node_id_list[i] != f.m_node_id_list[i])
        return false;
    }
    return m_node_id_list.size() < f.m_node_id_list.size();
  }

  PUGS_INLINE
  ConnectivityFace(const ConnectivityFace&) = default;

  PUGS_INLINE
  ConnectivityFace() = delete;

  PUGS_INLINE
  ~ConnectivityFace() = default;
};

#endif   // MESH_BUILDER_BASE_HPP
