#ifndef TABLE_HPP
#define TABLE_HPP

#include <Kokkos_CopyViews.hpp>
#include <utils/Array.hpp>
#include <utils/PugsAssert.hpp>
#include <utils/PugsMacros.hpp>
#include <utils/PugsUtils.hpp>

template <typename DataType>
class [[nodiscard]] Table
{
 public:
  using data_type  = DataType;
  using index_type = size_t;

 private:
  Kokkos::View<DataType**> m_values;

  // Allows const version to access our data
  friend Table<std::add_const_t<DataType>>;

 public:
  PUGS_INLINE size_t nbRows() const noexcept
  {
    return m_values.extent(0);
  }

  PUGS_INLINE size_t nbColumns() const noexcept
  {
    return m_values.extent(1);
  }

  PUGS_INLINE
  Array<DataType> operator[](index_type i) const
  {
    Assert(i < this->nbRows());
    return encapsulate(Kokkos::View<DataType*>(m_values, i, Kokkos::ALL));
  }

  friend PUGS_INLINE Table<std::remove_const_t<DataType>> copy(const Table<DataType>& source)
  {
    Table<std::remove_const_t<DataType>> image(source.nbRows(), source.nbColumns());
    Kokkos::deep_copy(image.m_values, source.m_values);

    return image;
  }

  friend PUGS_INLINE void copy_to(const Table<DataType>& source,
                                  const Table<std::remove_const_t<DataType>>& destination)
  {
    Assert(source.nbRows() == destination.nbRows());
    Assert(source.nbColumns() == destination.nbColumns());
    Kokkos::deep_copy(destination.m_values, source.m_values);
  }

  template <typename DataType2, typename... RT>
  friend PUGS_INLINE Table<DataType2> encapsulate(const Kokkos::View<DataType2**, RT...>& values);

  template <typename DataType2>
  friend PUGS_INLINE Table<DataType2> subTable(const Table<DataType2>& table,
                                               typename Table<DataType2>::index_type row_begin,
                                               typename Table<DataType2>::index_type row_size,
                                               typename Table<DataType2>::index_type column_begin,
                                               typename Table<DataType2>::index_type column_size);

  PUGS_INLINE DataType& operator()(index_type i, index_type j) const noexcept(NO_ASSERT)
  {
    Assert(i < this->nbRows());
    Assert(j < this->nbColumns());
    return m_values(i, j);
  }

  PUGS_INLINE
  void fill(const DataType& data) const
  {
    static_assert(not std::is_const<DataType>(), "Cannot modify Table of const");

    Kokkos::deep_copy(m_values, data);
  }

  template <typename DataType2>
  PUGS_INLINE Table& operator=(const Table<DataType2>& table) noexcept
  {
    // ensures that DataType is the same as source DataType2
    static_assert(std::is_same<std::remove_const_t<DataType>, std::remove_const_t<DataType2>>(),
                  "Cannot assign Table of different type");
    // ensures that const is not lost through copy
    static_assert(((std::is_const<DataType2>() and std::is_const<DataType>()) or not std::is_const<DataType2>()),
                  "Cannot assign Table of const to Table of non-const");
    m_values = table.m_values;
    return *this;
  }

  PUGS_INLINE
  Table& operator=(const Table&) = default;

  PUGS_INLINE
  Table& operator=(Table&&) = default;

  PUGS_INLINE
  explicit Table(size_t nb_lines, size_t nb_columns) : m_values("anonymous", nb_lines, nb_columns)
  {
    static_assert(not std::is_const<DataType>(), "Cannot allocate Table of const data: only view is "
                                                 "supported");
  }

  PUGS_INLINE
  Table() = default;

  PUGS_INLINE
  Table(const Table&) = default;

  template <typename DataType2>
  PUGS_INLINE Table(const Table<DataType2>& table) noexcept
  {
    this->operator=(table);
  }

  PUGS_INLINE
  Table(Table &&) = default;

  PUGS_INLINE
  ~Table() = default;
};

template <typename DataType, typename... RT>
PUGS_INLINE Table<DataType>
encapsulate(const Kokkos::View<DataType**, RT...>& values)
{
  Table<DataType> table;
  table.m_values = values;
  return table;
}

template <typename DataType>
PUGS_INLINE Table<DataType>
subTable(const Table<DataType>& table,
         typename Table<DataType>::index_type row_begin,
         typename Table<DataType>::index_type row_size,
         typename Table<DataType>::index_type column_begin,
         typename Table<DataType>::index_type column_size)
{
  Assert(row_begin < table.nbRows());
  Assert(row_begin + row_size <= table.nbRows());
  Assert(column_begin < table.nbColumns());
  Assert(column_begin + column_size <= table.nbColumns());
  return encapsulate(Kokkos::View<DataType**>(table.m_values, std::make_pair(row_begin, row_begin + row_size),
                                              std::make_pair(column_begin, column_begin + column_size)));
}

#endif   // TABLE_HPP
