#ifndef SYMBOL_TABLE_HPP
#define SYMBOL_TABLE_HPP

namespace language
{
class SymbolTable
{
  class Attributes
  {
    bool m_is_initialized{false};
    DataType m_data_type{DataType::undefined_t};
    DataVariant m_value;

   public:
    auto&
    value()
    {
      return m_value;
    }

    const auto&
    value() const
    {
      return m_value;
    }

    const bool&
    isInitialized() const
    {
      return m_is_initialized;
    }

    void
    setIsInitialized()
    {
      m_is_initialized = true;
    }

    const DataType&
    dataType() const
    {
      return m_data_type;
    }

    void
    setDataType(const DataType& data_type)
    {
      m_data_type = data_type;
    }

    friend std::ostream&
    operator<<(std::ostream& os, const Attributes& attributes)
    {
      std::visit(
        [&](const auto& value) {
          using T = std::decay_t<decltype(value)>;
          if constexpr (std::is_same_v<T, std::monostate>) {
            os << "--";
          } else {
            os << value;
          }
        },
        attributes.m_value);

      return os;
    }
  };

 private:
  std::vector<std::pair<std::string, Attributes>> m_symbol_list;
  std::shared_ptr<SymbolTable> m_parent_table;

 public:
  friend std::ostream&
  operator<<(std::ostream& os, const SymbolTable& symbol_table)
  {
    os << "-- Symbol table state -- parent : " << symbol_table.m_parent_table.get() << "\n";
    for (auto i_symbol : symbol_table.m_symbol_list) {
      os << ' ' << i_symbol.first << ": " << std::boolalpha << i_symbol.second << '\n';
    }
    os << "------------------------\n";
    return os;
  }

  auto
  find(const std::string& symbol)
  {
    auto i_symbol = m_symbol_list.end();

    for (auto i_stored_symbol = m_symbol_list.begin(); i_stored_symbol != m_symbol_list.end(); ++i_stored_symbol) {
      if (i_stored_symbol->first == symbol) {
        i_symbol = i_stored_symbol;
        break;
      }
    }

    if (i_symbol != m_symbol_list.end()) {
      return std::make_pair(i_symbol, true);
    } else {
      if (m_parent_table) {
        return m_parent_table->find(symbol);
      } else {
        return std::make_pair(i_symbol, false);
      }
    }
  }

  auto
  add(const std::string& symbol)
  {
    for (auto i_stored_symbol = m_symbol_list.begin(); i_stored_symbol != m_symbol_list.end(); ++i_stored_symbol) {
      if (i_stored_symbol->first == symbol) {
        return std::make_pair(i_stored_symbol, false);
      }
    }
    return std::make_pair(m_symbol_list.emplace(m_symbol_list.end(), std::make_pair(symbol, Attributes())), true);
  }

  SymbolTable(const std::shared_ptr<SymbolTable>& parent_table = nullptr) : m_parent_table(parent_table)
  {
    ;
  }
};
}   // namespace language

#endif   // SYMBOL_TABLE_HPP
