#include <language/modules/KineticSchemeModule.hpp>

#include <language/modules/MeshModuleTypes.hpp>
#include <language/modules/SchemeModuleTypes.hpp>

#include <language/modules/MeshModuleTypes.hpp>
#include <scheme/EulerKineticSolver.hpp>
#include <scheme/EulerKineticSolverAcoustic2LagrangeFVOrder2.hpp>
#include <scheme/EulerKineticSolverAcousticLagrangeFV.hpp>
#include <scheme/EulerKineticSolverFirstOrderFV.hpp>
#include <scheme/EulerKineticSolverLagrangeFV.hpp>
#include <scheme/EulerKineticSolverMeanFluxMood.hpp>
#include <scheme/EulerKineticSolverMoodFD.hpp>
#include <scheme/EulerKineticSolverMoodFV.hpp>
#include <scheme/EulerKineticSolverMultiD.hpp>
#include <scheme/EulerKineticSolverMultiDOrder2.hpp>
#include <scheme/EulerKineticSolverOneFluxMood.hpp>
#include <scheme/EulerKineticSolverThirdOrderFD.hpp>
#include <scheme/EulerKineticSolverThirdOrderFV.hpp>
#include <scheme/EulerKineticSolverThirdOrderMoodFD.hpp>
#include <scheme/EulerKineticSolverThirdOrderMoodFV.hpp>
#include <scheme/Scalar2WavesKineticSolver.hpp>
#include <scheme/Scalar3WavesKineticSolver.hpp>
#include <scheme/ScalarKineticSolver.hpp>
#include <scheme/WavesEquationKineticSolver.hpp>
#include <scheme/WavesEquationKineticSolverThirdOrderTimeFD.hpp>
#include <scheme/WavesEquationKineticSolverThirdOrderTimeFV.hpp>
#include <scheme/WavesEquationNonUniformCellKineticSolver.hpp>
#include <scheme/WavesEquationNonUniformKineticSolver.hpp>

#include <language/modules/ModuleRepository.hpp>
#include <language/utils/BuiltinFunctionEmbedder.hpp>

#include <memory>

KineticSchemeModule::KineticSchemeModule()
{
  this->_addBuiltinFunction("scalar_2waves_kinetic",
                            std::function(

                              [](const double& dt, const FunctionSymbolId& flux_function_id, const double& a,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F)
                                -> std::shared_ptr<const DiscreteFunctionVariant> {
                                return scalar2WavesKineticSolver(dt, flux_function_id, a, eps, F);
                              }

                              ));

  this->_addBuiltinFunction("scalar_3waves_kinetic",
                            std::function(

                              [](const double& dt, const FunctionSymbolId& flux_function_id, const double& a,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F)
                                -> std::shared_ptr<const DiscreteFunctionVariant> {
                                return scalar3WavesKineticSolver(dt, flux_function_id, a, eps, F);
                              }

                              ));

  this->_addBuiltinFunction("scalar_kinetic", std::function(

                                                [](const double& dt, const FunctionSymbolId& flux_function_id,
                                                   const std::vector<double>& lambda_vector, const double& eps,
                                                   const std::shared_ptr<const DiscreteFunctionVariant>& F)
                                                  -> std::shared_ptr<const DiscreteFunctionVariant> {
                                                  return scalarKineticSolver(dt, flux_function_id, lambda_vector, eps,
                                                                             F);
                                                }

                                                ));

  this->_addBuiltinFunction("get_scalar_kinetic_waves", std::function(

                                                          [](const std::vector<double>& lambda_vector,
                                                             const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                                             const std::shared_ptr<const DiscreteFunctionVariant>& Au)
                                                            -> std::shared_ptr<const DiscreteFunctionVariant> {
                                                            return getScalarKineticWaves(lambda_vector, u, Au);
                                                          }));
  this->_addBuiltinFunction("waves_equation_kinetic",
                            std::function(

                              [](const double& dt, const TinyMatrix<2>& A, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_p)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>, double> {
                                return wavesEquationKineticSolver(dt, A, lambda_vector, eps, F_u, F_p);
                              }

                              ));

  this->_addBuiltinFunction("get_waves_equation_kinetic_waves",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& p,
                                 const TinyMatrix<2>& A) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                       std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getWavesEquationKineticWaves(lambda_vector, u, p, A);
                              }

                              ));

  this->_addBuiltinFunction("waves_equation_kinetic_third_order_time_FV",
                            std::function(

                              [](const double& dt, const TinyMatrix<2>& A, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_p)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>, double> {
                                return wavesEquationKineticSolverThirdOrderTimeFV(dt, A, lambda_vector, eps, F_u, F_p);
                              }

                              ));

  this->_addBuiltinFunction("get_waves_equation_kinetic_waves_third_order_time_FV",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& p,
                                 const TinyMatrix<2>& A) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                       std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getWavesEquationKineticWavesThirdOrderTimeFV(lambda_vector, u, p, A);
                              }

                              ));

  this->_addBuiltinFunction("waves_equation_kinetic_third_order_time_FD",
                            std::function(

                              [](const double& dt, const TinyMatrix<2>& A, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_p)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>, double> {
                                return wavesEquationKineticSolverThirdOrderTimeFD(dt, A, lambda_vector, eps, F_u, F_p);
                              }

                              ));

  this->_addBuiltinFunction("get_waves_equation_kinetic_waves_third_order_time_FD",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& p,
                                 const TinyMatrix<2>& A) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                       std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getWavesEquationKineticWavesThirdOrderTimeFV(lambda_vector, u, p, A);
                              }

                              ));

  this->_addBuiltinFunction("waves_equation_non_uniform_kinetic",
                            std::function(

                              [](const double& dt, const TinyMatrix<2>& A,
                                 const std::shared_ptr<const ItemArrayVariant>& lambda_vector, const double& eps,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_p)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>, double> {
                                return wavesEquationNonUniformKineticSolver(dt, A, lambda_vector, eps, F_u, F_p);
                              }

                              ));

  this->_addBuiltinFunction("get_waves_equation_non_uniform_kinetic_waves",
                            std::function(

                              [](const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& p, const TinyMatrix<2>& A,
                                 const uint64_t& nb_waves,
                                 const double& dt) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                 std::shared_ptr<const DiscreteFunctionVariant>,
                                                                 std::shared_ptr<const ItemArrayVariant>> {
                                return getWavesEquationNonUniformKineticWaves(u, p, A, nb_waves, dt);
                              }

                              ));

  this->_addBuiltinFunction("waves_equation_non_uniform_cell_kinetic",
                            std::function(

                              [](const double& dt, const TinyMatrix<2>& A,
                                 const std::shared_ptr<const ItemArrayVariant>& lambda_vector, const double& eps,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_p)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>, double> {
                                return wavesEquationNonUniformCellKineticSolver(dt, A, lambda_vector, eps, F_u, F_p);
                              }

                              ));

  this->_addBuiltinFunction("get_waves_equation_non_uniform_cell_kinetic_waves",
                            std::function(

                              [](const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& p, const TinyMatrix<2>& A,
                                 const uint64_t& nb_waves,
                                 const double& dt) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                 std::shared_ptr<const DiscreteFunctionVariant>,
                                                                 std::shared_ptr<const ItemArrayVariant>> {
                                return getWavesEquationNonUniformCellKineticWaves(u, p, A, nb_waves, dt);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolver(dt, gamma, lambda_vector, eps, F_rho, F_rho_u, F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWaves(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution",
                            std::function(
                              [](const double& t, const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u_ex)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticExactSolution(t, rho_ex, u_ex);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_mean_flux_mood",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverMeanFluxMood(dt, gamma, lambda_vector, eps, F_rho, F_rho_u,
                                                                      F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_mean_flux_mood",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesMeanFluxMood(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_mean_flux_mood",
                            std::function(
                              [](const double& t, const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u_ex)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticExactSolutionMeanFluxMood(t, rho_ex, u_ex);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_one_flux_mood",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverOneFluxMood(dt, gamma, lambda_vector, eps, F_rho, F_rho_u,
                                                                     F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_one_flux_mood",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesOneFluxMood(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_one_flux_mood",
                            std::function(
                              [](const double& t, const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u_ex)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticExactSolutionOneFluxMood(t, rho_ex, u_ex);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_mood_FV",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverMoodFV(dt, gamma, lambda_vector, eps, F_rho, F_rho_u, F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_mood_FV",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesMoodFV(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_mood_FV",
                            std::function(
                              [](const double& t, const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u_ex)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticExactSolutionMoodFV(t, rho_ex, u_ex);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_mood_FD",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverMoodFD(dt, gamma, lambda_vector, eps, F_rho, F_rho_u, F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_mood_FD",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesMoodFD(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_mood_FD",
                            std::function(
                              [](const double& t, const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u_ex)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticExactSolutionMoodFD(t, rho_ex, u_ex);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_third_order_mood_FV",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverThirdOrderMoodFV(dt, gamma, lambda_vector, eps, F_rho, F_rho_u,
                                                                          F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_third_order_mood_FV",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesThirdOrderMoodFV(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_third_order_mood_FV",
                            std::function([](const double& t, const double& gamma,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& p_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_E_ex)
                                            -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>> {
                              return getEulerKineticExactSolutionThirdOrderMoodFV(t, gamma, rho_ex, u_ex, p_ex,
                                                                                  rho_u_ex, rho_E_ex);
                            }

                                          ));

  this->_addBuiltinFunction("euler_kinetic_third_order_mood_FD",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverThirdOrderMoodFD(dt, gamma, lambda_vector, eps, F_rho, F_rho_u,
                                                                          F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_third_order_mood_FD",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesThirdOrderMoodFD(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_third_order_mood_FD",
                            std::function(
                              [](const double& t, const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u_ex)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticExactSolutionThirdOrderMoodFD(t, rho_ex, u_ex);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_third_order_FV",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const size_t& SpaceOrder, const size_t& Limitation,
                                 const bool& MoodLimitation,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverThirdOrderFV(dt, gamma, lambda_vector, eps, SpaceOrder,
                                                                      Limitation, MoodLimitation, F_rho, F_rho_u, F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_third_order_FV",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesThirdOrderFV(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_third_order_FV",
                            std::function([](const double& t, const double& gamma,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& p_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_E_ex)
                                            -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>> {
                              return getEulerKineticExactSolutionThirdOrderFV(t, gamma, rho_ex, u_ex, p_ex, rho_u_ex,
                                                                              rho_E_ex);
                            }

                                          ));

  this->_addBuiltinFunction("euler_kinetic_MultiD",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<TinyVector<2>>& lambda_vector,
                                 const double& eps, const size_t& SpaceOrder, const size_t& scheme,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E,
                                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>&
                                   bc_descriptor_list) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverMultiD(dt, gamma, lambda_vector, eps, SpaceOrder, scheme,
                                                                F_rho, F_rho_u, F_E, bc_descriptor_list);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_MultiD",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<TinyVector<3>>& lambda_vector,
                                 const double& eps, const size_t& SpaceOrder, const size_t& scheme,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E,
                                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>&
                                   bc_descriptor_list) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverMultiD(dt, gamma, lambda_vector, eps, SpaceOrder, scheme,
                                                                F_rho, F_rho_u, F_E, bc_descriptor_list);
                              }

                              ));

  this->_addBuiltinFunction("get_lambda_vector_2d",
                            std::function(

                              [](const std::shared_ptr<const MeshVariant>& mesh, const double& lambda, const size_t& L,
                                 const size_t& M) -> std::vector<TinyVector<2>> {
                                return getLambdaVector2D(mesh, lambda, L, M);
                              }

                              ));

  this->_addBuiltinFunction("get_lambda_vector_3d", std::function(

                                                      [](const std::shared_ptr<const MeshVariant>& mesh,
                                                         const double& lambda) -> std::vector<TinyVector<3>> {
                                                        return getLambdaVector3D(mesh, lambda);
                                                      }

                                                      ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_MultiD",
                            std::function(

                              [](const std::vector<TinyVector<2>>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesMultiD(lambda_vector, rho, rho_u, rho_E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_MultiD",
                            std::function(

                              [](const std::vector<TinyVector<3>>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesMultiD(lambda_vector, rho, rho_u, rho_E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_MultiD_Order2",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<TinyVector<2>>& lambda_vector,
                                 const double& eps, const size_t& SpaceOrder, const size_t& TimeOrder,
                                 const size_t& scheme, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E,
                                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>&
                                   bc_descriptor_list) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverMultiDOrder2(dt, gamma, lambda_vector, eps, SpaceOrder,
                                                                      TimeOrder, scheme, F_rho, F_rho_u, F_E,
                                                                      bc_descriptor_list);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_MultiD_Order2",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<TinyVector<3>>& lambda_vector,
                                 const double& eps, const size_t& SpaceOrder, const size_t& TimeOrder,
                                 const size_t& scheme, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E,
                                 const std::vector<std::shared_ptr<const IBoundaryConditionDescriptor>>&
                                   bc_descriptor_list) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>,
                                                                     std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverMultiDOrder2(dt, gamma, lambda_vector, eps, SpaceOrder,
                                                                      TimeOrder, scheme, F_rho, F_rho_u, F_E,
                                                                      bc_descriptor_list);
                              }

                              ));

  this->_addBuiltinFunction("euler_kinetic_first_order_FV",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const size_t& SpaceOrder, const size_t& Limitation,
                                 const bool& MoodLimitation,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverFirstOrderFV(dt, gamma, lambda_vector, eps, SpaceOrder,
                                                                      Limitation, MoodLimitation, F_rho, F_rho_u, F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_first_order_FV",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesFirstOrderFV(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_first_order_FV",
                            std::function([](const double& t, const double& gamma,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& p_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_E_ex)
                                            -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>> {
                              return getEulerKineticExactSolutionFirstOrderFV(t, gamma, rho_ex, u_ex, p_ex, rho_u_ex,
                                                                              rho_E_ex);
                            }

                                          ));

  this->_addBuiltinFunction("refine",
                            std::function([](const std::shared_ptr<const DiscreteFunctionVariant>& old_u_old_mesh,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& new_u_new_mesh)
                                            -> std::shared_ptr<const DiscreteFunctionVariant> {
                              return refine(old_u_old_mesh, new_u_new_mesh);
                            }));

  this->_addBuiltinFunction("coarsen",
                            std::function([](const std::shared_ptr<const DiscreteFunctionVariant>& u_old_mesh,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& new_u_new_mesh)
                                            -> std::shared_ptr<const DiscreteFunctionVariant> {
                              return coarsen(u_old_mesh, new_u_new_mesh);
                            }));

  this->_addBuiltinFunction("euler_kinetic_lagrange_FV",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const size_t& space_order, const size_t& time_order, const double& eps,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const MeshVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverLagrangeFV(dt, gamma, lambda_vector, space_order, time_order,
                                                                    eps, F_rho, F_rho_u, F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_lagrange_FV",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesLagrangeFV(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_lagrange_FV",
                            std::function([](const double& t, const double& gamma,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& p_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_E_ex)
                                            -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>> {
                              return getEulerKineticExactSolutionLagrangeFV(t, gamma, rho_ex, u_ex, p_ex, rho_u_ex,
                                                                            rho_E_ex);
                            }

                                          ));

  this->_addBuiltinFunction("euler_kinetic_acoustic_lagrange_FV",
                            std::function(

                              [](const double& dt, const std::vector<double>& lambda_vector, const double& gamma,
                                 const double& eps, const size_t& space_order, const size_t& time_order,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& p)
                                -> std::tuple<std::shared_ptr<const MeshVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverAcousticLagrangeFV(dt, lambda_vector, gamma, eps, space_order,
                                                                            time_order, rho, u, E, p);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_acoustic_lagrange_FV",
                            std::function([](const double& t, const double& gamma,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& p_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_E_ex)
                                            -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>> {
                              return getEulerKineticExactSolutionAcousticLagrangeFV(t, gamma, rho_ex, u_ex, p_ex,
                                                                                    rho_u_ex, rho_E_ex);
                            }

                                          ));

  this->_addBuiltinFunction("euler_kinetic_acoustic2_lagrange_FV_order2",
                            std::function([](const double& dt, const std::vector<double>& lambda_vector,
                                             const double& gamma, const double& eps, const size_t& space_order,
                                             const size_t& time_order, const bool& TVD_limitation,
                                             const bool& MOOD_limitation,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& u,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& p)
                                            -> std::tuple<std::shared_ptr<const MeshVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>> {
                              return eulerKineticSolverAcoustic2LagrangeFVOrder2(dt, lambda_vector, gamma, eps,
                                                                                 space_order, time_order,
                                                                                 TVD_limitation, MOOD_limitation, rho,
                                                                                 u, E, p);
                            }));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_acoustic2_lagrange_FV_order2",
                            std::function([](const double& t, const double& gamma,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& p_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_u_ex,
                                             const std::shared_ptr<const DiscreteFunctionVariant>& rho_E_ex)
                                            -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>,
                                                          std::shared_ptr<const DiscreteFunctionVariant>> {
                              return getEulerKineticExactSolutionAcoustic2LagrangeFVOrder2(t, gamma, rho_ex, u_ex, p_ex,
                                                                                           rho_u_ex, rho_E_ex);
                            }

                                          ));

  this->_addBuiltinFunction("euler_kinetic_third_order_FD",
                            std::function(

                              [](const double& dt, const double& gamma, const std::vector<double>& lambda_vector,
                                 const double& eps, const std::shared_ptr<const DiscreteFunctionVariant>& F_rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& F_E)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return eulerKineticSolverThirdOrderFD(dt, gamma, lambda_vector, eps, F_rho, F_rho_u,
                                                                      F_E);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_waves_third_order_FD",
                            std::function(

                              [](const std::vector<double>& lambda_vector,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& rho_u,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& E,
                                 const double& gamma) -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>,
                                                                    std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticWavesThirdOrderFD(lambda_vector, rho, rho_u, E, gamma);
                              }

                              ));

  this->_addBuiltinFunction("get_euler_kinetic_exact_solution_third_order_FD",
                            std::function(
                              [](const double& t, const std::shared_ptr<const DiscreteFunctionVariant>& rho_ex,
                                 const std::shared_ptr<const DiscreteFunctionVariant>& u_ex)
                                -> std::tuple<std::shared_ptr<const DiscreteFunctionVariant>,
                                              std::shared_ptr<const DiscreteFunctionVariant>> {
                                return getEulerKineticExactSolutionThirdOrderFD(t, rho_ex, u_ex);
                              }

                              ));
}

void
KineticSchemeModule::registerOperators() const
{}

void
KineticSchemeModule::registerCheckpointResume() const
{}

ModuleRepository::Subscribe<KineticSchemeModule> kinetic_scheme_module;
