#ifndef MESH_HPP
#define MESH_HPP

#include <ItemValue.hpp>
#include <TinyVector.hpp>

#include <CSRGraph.hpp>

#include <memory>

struct IMesh
{
  virtual size_t dimension() const = 0;
  ~IMesh()                         = default;
};

template <typename ConnectivityType>
class Mesh final : public IMesh
{
 public:
  using Connectivity = ConnectivityType;

  static constexpr size_t Dimension = ConnectivityType::Dimension;
  using Rd                          = TinyVector<Dimension>;

 private:
  const std::shared_ptr<const Connectivity> m_connectivity;
  NodeValue<const Rd> m_xr;
  NodeValue<Rd> m_mutable_xr;

 public:
  PUGS_INLINE
  size_t
  dimension() const
  {
    return Dimension;
  }

  PUGS_INLINE
  const Connectivity&
  connectivity() const
  {
    return *m_connectivity;
  }

  PUGS_INLINE
  size_t
  numberOfNodes() const
  {
    return m_connectivity->numberOfNodes();
  }

  PUGS_INLINE
  size_t
  numberOfFaces() const
  {
    return m_connectivity->numberOfFaces();
  }

  PUGS_INLINE
  size_t
  numberOfCells() const
  {
    return m_connectivity->numberOfCells();
  }

  PUGS_INLINE
  const NodeValue<const Rd>&
  xr() const
  {
    return m_xr;
  }

  PUGS_INLINE
  NodeValue<Rd>
  mutableXr() const
  {
    return m_mutable_xr;
  }

  PUGS_INLINE
  Mesh(const std::shared_ptr<const Connectivity>& connectivity, NodeValue<Rd>& xr)
    : m_connectivity{connectivity}, m_xr{xr}, m_mutable_xr{xr}
  {
    ;
  }

  Mesh() = delete;

  ~Mesh()
  {
    ;
  }
};

#endif   // MESH_HPP
