#include <GmshReader.hpp>
#include <PugsMacros.hpp>

#include <fstream>
#include <iostream>
#include <rang.hpp>
#include <set>

#include <CellType.hpp>
#include <Connectivity.hpp>

#include <Mesh.hpp>
#include <MeshData.hpp>

#include <Messenger.hpp>
#include <RefItemList.hpp>

#include <ArrayUtils.hpp>
#include <ItemValueUtils.hpp>

#include <ConnectivityDispatcher.hpp>

#include <iomanip>
#include <iostream>

#include <map>
#include <regex>
#include <unordered_map>

class ErrorHandler
{
 public:
  enum Type
  {
    asked,       /**< execution request by the user*/
    compilation, /**< syntax error in a language */
    normal,      /**< normal error due to a bad use of ff3d */
    unexpected   /**< Unexpected execution error */
  };

 private:
  const std::string __filename;     /**< The source file name where the error occured */
  const size_t __lineNumber;        /**< The line number where exception was raised */
  const std::string __errorMessage; /**< The reporting message */

  const Type __type; /**< the type of the error */
 public:
  /**
   * Prints the error message
   *
   */
  virtual void writeErrorMessage() const;

  /**
   * The copy constructor
   *
   * @param e an handled error
   */
  ErrorHandler(const ErrorHandler& e)
    : __filename(e.__filename), __lineNumber(e.__lineNumber), __errorMessage(e.__errorMessage), __type(e.__type)
  {
    ;
  }

  /**
   * Constructor
   *
   * @param filename the source file name
   * @param lineNumber the source line
   * @param errorMessage the reported message
   * @param type the type of the error
   */
  ErrorHandler(const std::string& filename,
               const size_t& lineNumber,
               const std::string& errorMessage,
               const Type& type);

  /**
   * The destructor
   *
   */
  virtual ~ErrorHandler()
  {
    ;
  }
};

void
ErrorHandler::writeErrorMessage() const
{
  switch (__type) {
  case asked: {
    std::cerr << "\nremark: exit command explicitly called\n";
    [[fallthrough]];
  }
  case normal: {
    std::cerr << '\n' << __filename << ':' << __lineNumber << ":remark: emitted the following message\n";
    std::cerr << "error: " << __errorMessage << '\n';
    break;
  }
  case compilation: {
    std::cerr << "\nline " << __lineNumber << ':' << __errorMessage << '\n';
    break;
  }
  case unexpected: {
    std::cerr << '\n' << __filename << ':' << __lineNumber << ":\n" << __errorMessage << '\n';
    std::cerr << "\nUNEXPECTED ERROR: this should not occure, please report it\n";
    std::cerr << "\nBUG REPORT: Please send bug reports to:\n"
              << "  ff3d-dev@nongnu.org or freefem@ann.jussieu.fr\n"
              << "or better, use the Bug Tracking System:\n"
              << "  http://savannah.nongnu.org/bugs/?group=ff3d\n";
    break;
  }
  default: {
    std::cerr << __filename << ':' << __lineNumber << ": " << __errorMessage << '\n';
    std::cerr << __FILE__ << ':' << __LINE__ << ":remark:  error type not implemented!\n";
  }
  }
}

ErrorHandler::ErrorHandler(const std::string& filename,
                           const size_t& lineNumber,
                           const std::string& errorMessage,
                           const Type& type)
  : __filename(filename), __lineNumber(lineNumber), __errorMessage(errorMessage), __type(type)
{
  ;
}

template <int Dimension>
void
GmshReader::_dispatch()
{
  using ConnectivityType = Connectivity<Dimension>;
  using Rd               = TinyVector<Dimension>;
  using MeshType         = Mesh<ConnectivityType>;

  if (not m_mesh) {
    ConnectivityDescriptor descriptor;
    std::shared_ptr connectivity = ConnectivityType::build(descriptor);
    NodeValue<Rd> xr;
    m_mesh = std::make_shared<MeshType>(connectivity, xr);
  }
  const MeshType& mesh = static_cast<const MeshType&>(*m_mesh);

  ConnectivityDispatcher<Dimension> dispatcher(mesh.connectivity());

  std::shared_ptr dispatched_connectivity = dispatcher.dispatchedConnectivity();
  NodeValue<Rd> dispatched_xr             = dispatcher.dispatch(mesh.xr());

  m_mesh = std::make_shared<MeshType>(dispatched_connectivity, dispatched_xr);
}

template <size_t Dimension>
class ConnectivityFace;

template <>
class ConnectivityFace<2>
{
 public:
  friend struct Hash;
  struct Hash
  {
    size_t
    operator()(const ConnectivityFace& f) const
    {
      size_t hash = 0;
      hash ^= std::hash<unsigned int>()(f.m_node0_id);
      hash ^= std::hash<unsigned int>()(f.m_node1_id) >> 1;
      return hash;
    }
  };

 private:
  const std::vector<int>& m_node_number_vector;

  unsigned int m_node0_id;
  unsigned int m_node1_id;

  bool m_reversed;

 public:
  std::vector<unsigned int>
  nodeIdList() const
  {
    return {m_node0_id, m_node1_id};
  }

  bool
  reversed() const
  {
    return m_reversed;
  }

  PUGS_INLINE
  bool
  operator==(const ConnectivityFace& f) const
  {
    return ((m_node0_id == f.m_node0_id) and (m_node1_id == f.m_node1_id));
  }

  PUGS_INLINE
  bool
  operator<(const ConnectivityFace& f) const
  {
    return ((m_node_number_vector[m_node0_id] < m_node_number_vector[f.m_node0_id]) or
            ((m_node_number_vector[m_node0_id] == m_node_number_vector[f.m_node0_id]) and
             (m_node_number_vector[m_node1_id] < m_node_number_vector[f.m_node1_id])));
  }

  PUGS_INLINE
  ConnectivityFace(const std::vector<unsigned int>& node_id_list, const std::vector<int>& node_number_vector)
    : m_node_number_vector(node_number_vector)
  {
    Assert(node_id_list.size() == 2);

    if (m_node_number_vector[node_id_list[0]] < m_node_number_vector[node_id_list[1]]) {
      m_node0_id = node_id_list[0];
      m_node1_id = node_id_list[1];
      m_reversed = false;
    } else {
      m_node0_id = node_id_list[1];
      m_node1_id = node_id_list[0];
      m_reversed = true;
    }
  }

  PUGS_INLINE
  ConnectivityFace(const ConnectivityFace&) = default;

  PUGS_INLINE
  ~ConnectivityFace() = default;
};

template <>
class ConnectivityFace<3>
{
 private:
  friend class GmshReader;
  friend struct Hash;
  struct Hash
  {
    size_t
    operator()(const ConnectivityFace& f) const
    {
      size_t hash = 0;
      for (size_t i = 0; i < f.m_node_id_list.size(); ++i) {
        hash ^= std::hash<unsigned int>()(f.m_node_id_list[i]) >> i;
      }
      return hash;
    }
  };

 private:
  bool m_reversed;
  std::vector<NodeId::base_type> m_node_id_list;
  const std::vector<int>& m_node_number_vector;

  PUGS_INLINE
  std::vector<unsigned int>
  _sort(const std::vector<unsigned int>& node_list)
  {
    const auto min_id = std::min_element(node_list.begin(), node_list.end());
    const int shift   = std::distance(node_list.begin(), min_id);

    std::vector<unsigned int> rotated_node_list(node_list.size());
    if (node_list[(shift + 1) % node_list.size()] > node_list[(shift + node_list.size() - 1) % node_list.size()]) {
      for (size_t i = 0; i < node_list.size(); ++i) {
        rotated_node_list[i] = node_list[(shift + node_list.size() - i) % node_list.size()];
        m_reversed           = true;
      }
    } else {
      for (size_t i = 0; i < node_list.size(); ++i) {
        rotated_node_list[i] = node_list[(shift + i) % node_list.size()];
      }
    }

    return rotated_node_list;
  }

 public:
  PUGS_INLINE
  const bool&
  reversed() const
  {
    return m_reversed;
  }

  PUGS_INLINE
  const std::vector<unsigned int>&
  nodeIdList() const
  {
    return m_node_id_list;
  }

  PUGS_INLINE
  ConnectivityFace(const std::vector<unsigned int>& given_node_id_list, const std::vector<int>& node_number_vector)
    : m_reversed(false), m_node_id_list(_sort(given_node_id_list)), m_node_number_vector(node_number_vector)
  {
    ;
  }

 public:
  bool
  operator==(const ConnectivityFace& f) const
  {
    if (m_node_id_list.size() == f.nodeIdList().size()) {
      for (size_t j = 0; j < m_node_id_list.size(); ++j) {
        if (m_node_id_list[j] != f.nodeIdList()[j]) {
          return false;
        }
      }
      return true;
    }
    return false;
  }

  PUGS_INLINE
  bool
  operator<(const ConnectivityFace& f) const
  {
    const size_t min_nb_nodes = std::min(f.m_node_id_list.size(), m_node_id_list.size());
    for (size_t i = 0; i < min_nb_nodes; ++i) {
      if (m_node_id_list[i] < f.m_node_id_list[i])
        return true;
      if (m_node_id_list[i] != f.m_node_id_list[i])
        return false;
    }
    return m_node_id_list.size() < f.m_node_id_list.size();
  }

  PUGS_INLINE
  ConnectivityFace(const ConnectivityFace&) = default;

  PUGS_INLINE
  ConnectivityFace() = delete;

  PUGS_INLINE
  ~ConnectivityFace() = default;
};

GmshReader::GmshReader(const std::string& filename) : m_filename(filename)
{
  if (parallel::rank() == 0) {
    try {
      m_fin.open(m_filename);
      if (not m_fin) {
        std::cerr << "cannot read file '" << m_filename << "'\n";
        std::exit(0);
      }

      // gmsh 2.2 format keywords
      __keywordList["$MeshFormat"]       = MESHFORMAT;
      __keywordList["$EndMeshFormat"]    = ENDMESHFORMAT;
      __keywordList["$Nodes"]            = NODES;
      __keywordList["$EndNodes"]         = ENDNODES;
      __keywordList["$Elements"]         = ELEMENTS;
      __keywordList["$EndElements"]      = ENDELEMENTS;
      __keywordList["$PhysicalNames"]    = PHYSICALNAMES;
      __keywordList["$EndPhysicalNames"] = ENDPHYSICALNAMES;

      __numberOfPrimitiveNodes.resize(16);
      __numberOfPrimitiveNodes[0]  = 2;    // edge
      __numberOfPrimitiveNodes[1]  = 3;    // triangle
      __numberOfPrimitiveNodes[2]  = 4;    // quadrangle
      __numberOfPrimitiveNodes[3]  = 4;    // Tetrahedron
      __numberOfPrimitiveNodes[4]  = 8;    // Hexaredron
      __numberOfPrimitiveNodes[5]  = 6;    // Prism
      __numberOfPrimitiveNodes[6]  = 5;    // Pyramid
      __numberOfPrimitiveNodes[7]  = 3;    // second order edge
      __numberOfPrimitiveNodes[8]  = 6;    // second order triangle
      __numberOfPrimitiveNodes[9]  = 9;    // second order quadrangle
      __numberOfPrimitiveNodes[10] = 10;   // second order tetrahedron
      __numberOfPrimitiveNodes[11] = 27;   // second order hexahedron
      __numberOfPrimitiveNodes[12] = 18;   // second order prism
      __numberOfPrimitiveNodes[13] = 14;   // second order pyramid
      __numberOfPrimitiveNodes[14] = 1;    // point

      __primitivesNames[0]      = "edges";
      __supportedPrimitives[0]  = true;
      __primitivesNames[1]      = "triangles";
      __supportedPrimitives[1]  = true;
      __primitivesNames[2]      = "quadrangles";
      __supportedPrimitives[2]  = true;
      __primitivesNames[3]      = "tetrahedra";
      __supportedPrimitives[3]  = true;
      __primitivesNames[4]      = "hexahedra";
      __supportedPrimitives[4]  = true;
      __primitivesNames[5]      = "prisms";
      __supportedPrimitives[5]  = false;
      __primitivesNames[6]      = "pyramids";
      __supportedPrimitives[6]  = false;
      __primitivesNames[7]      = "second order edges";
      __supportedPrimitives[7]  = false;
      __primitivesNames[8]      = "second order triangles";
      __supportedPrimitives[8]  = false;
      __primitivesNames[9]      = "second order quadrangles";
      __supportedPrimitives[9]  = false;
      __primitivesNames[10]     = "second order tetrahedra";
      __supportedPrimitives[10] = false;
      __primitivesNames[11]     = "second order hexahedra";
      __supportedPrimitives[11] = false;
      __primitivesNames[12]     = "second order prisms";
      __supportedPrimitives[12] = false;
      __primitivesNames[13]     = "second order pyramids";
      __supportedPrimitives[13] = false;
      __primitivesNames[14]     = "point";
      __supportedPrimitives[14] = true;

      std::cout << "Reading file '" << m_filename << "'\n";

      // Getting vertices list
      GmshReader::Keyword kw = this->__nextKeyword();
      switch (kw.second) {
      // case NOD: {
      //   this->__readGmsh1();
      //   break;
      // }
      case MESHFORMAT: {
        double fileVersion = this->_getReal();
        if (fileVersion != 2.2) {
          throw ErrorHandler(__FILE__, __LINE__, "Cannot read Gmsh format '" + std::to_string(fileVersion) + "'",
                             ErrorHandler::normal);
        }
        int fileType = this->_getInteger();
        __binary     = (fileType == 1);
        if ((fileType < 0) or (fileType > 1)) {
          throw ErrorHandler(__FILE__, __LINE__, "Cannot read Gmsh file type '" + std::to_string(fileType) + "'",
                             ErrorHandler::normal);
        }

        int dataSize = this->_getInteger();
        if (dataSize != sizeof(double)) {
          throw ErrorHandler(__FILE__, __LINE__, "Data size not supported '" + std::to_string(dataSize) + "'",
                             ErrorHandler::normal);
        }

        if (__binary) {
          //       int one=0;

          //       fseek(__ifh,1L,SEEK_CUR);
          //       fread(reinterpret_cast<char*>(&one),sizeof(int),1,__ifh);

          //       if (one == 1) {
          // 	__convertEndian = false;
          //       } else {
          // 	invertEndianess(one);

          // 	if (one == 1) {
          // 	  __convertEndian = true;
          // 	} else {
          // 	  throw ErrorHandler(__FILE__,__LINE__,
          // 	  		     "Cannot determine endianess",
          // 	  		     ErrorHandler::normal);
          // 	}
          // }

          //       std::cout << "- Binary format: ";
          // #ifdef    WORDS_BIGENDIAN
          //       if (not __convertEndian) {
          // 	std::cout << "big endian\n";
          //       } else {
          // 	std::cout << "little endian\n";
          //       }
          // #else  // WORDS_BIGENDIAN
          //       if (not __convertEndian) {
          // 	std::cout << "little endian\n";
          //       } else {
          // 	std::cout << "big endian\n";
          //       }
          // #endif // WORDS_BIGENDIAN
        }

        kw = this->__nextKeyword();
        if (kw.second != ENDMESHFORMAT) {
          throw ErrorHandler(__FILE__, __LINE__,
                             "reading file '" + m_filename + "': expecting $EndMeshFormat, '" + kw.first +
                               "' was found",
                             ErrorHandler::normal);
        }

        this->__readGmshFormat2_2();

        break;
      }
      default: {
        throw ErrorHandler(__FILE__, __LINE__, "cannot determine format version of '" + m_filename + "'",
                           ErrorHandler::normal);
      }
      }

      this->__proceedData();
      // this->__createMesh();
    }
    catch (const ErrorHandler& e) {
      e.writeErrorMessage();
      std::exit(0);
    }
  }
  std::cout << std::flush;
  if (parallel::size() > 1) {
    std::cout << "Sequential mesh read! Need to be dispatched\n" << std::flush;

    const int mesh_dimension = [&]() {
      int mesh_dimension = -1;   // unknown mesh dimension
      if (m_mesh) {
        mesh_dimension = m_mesh->dimension();
      }

      Array<int> dimensions = parallel::allGather(mesh_dimension);
      std::set<int> dimension_set;
      for (size_t i = 0; i < dimensions.size(); ++i) {
        const int i_dimension = dimensions[i];
        if (i_dimension != -1) {
          dimension_set.insert(i_dimension);
        }
      }
      if (dimension_set.size() != 1) {
        std::cerr << "error dimensions of read mesh parts differ!\n";
        std::exit(1);
      }

      return *begin(dimension_set);
    }();
    switch (mesh_dimension) {
    case 1: {
      this->_dispatch<1>();
      break;
    }
    case 2: {
      this->_dispatch<2>();
      break;
    }
    case 3: {
      this->_dispatch<3>();
      break;
    }
    default: {
      std::cerr << "unexpected dimension " << mesh_dimension << '\n';
      std::exit(1);
    }
    }
  }
}

void
GmshReader::__readVertices()
{
  const int numberOfVerices = this->_getInteger();
  std::cout << "- Number Of Vertices: " << numberOfVerices << '\n';
  if (numberOfVerices < 0) {
    throw ErrorHandler(__FILE__, __LINE__, "reading file '" + this->m_filename + "': number of vertices is negative",
                       ErrorHandler::normal);
  }

  __verticesNumbers.resize(numberOfVerices);
  __vertices = Array<R3>(numberOfVerices);

  if (not __binary) {
    for (int i = 0; i < numberOfVerices; ++i) {
      __verticesNumbers[i] = this->_getInteger();
      const double x       = this->_getReal();
      const double y       = this->_getReal();
      const double z       = this->_getReal();
      __vertices[i]        = TinyVector<3, double>(x, y, z);
    }
  } else {
    // fseek(m_fin.file()__ifh,1L,SEEK_CUR);
    // for (int i=0; i<numberOfVerices; ++i) {
    //   int number = 0;
    //   fread(reinterpret_cast<char*>(&number),sizeof(int),1,__ifh);
    //   __verticesNumbers[i] = number;
    //   TinyVector<3,double> x;
    //   fread(reinterpret_cast<char*>(&(x[0])),sizeof(double),3,__ifh);
    //   for (size_t j=0; j<3; ++j) {
    // 	__vertices[i][j] = x[j];
    //   }
    // }

    // if (__convertEndian) {
    //   for (int i=0; i<numberOfVerices; ++i) {
    // 	invertEndianess(__verticesNumbers[i]);
    // 	for (size_t j=0; j<3; ++j) {
    // 	  invertEndianess(vertices[i][j]);
    // 	}
    //   }
    // }
  }
}

// void
// GmshReader::__readElements1()
// {
//   const int numberOfElements = this->_getInteger();
//   std::cout << "- Number Of Elements: " << numberOfElements << '\n';
//   if (numberOfElements<0) {
//     throw ErrorHandler(__FILE__,__LINE__,
// 		       "reading file '"+m_filename
// 		       +"': number of elements is negative",
// 		       ErrorHandler::normal);
//   }

//   __elementType.resize(numberOfElements);
//   __references.resize(numberOfElements);
//   __elementVertices.resize(numberOfElements);

//   for (int i=0; i<numberOfElements; ++i) {
//     this->_getInteger(); // drops element number
//     const int elementType = this->_getInteger()-1;

//     if ((elementType < 0) or (elementType > 14)) {
//       throw ErrorHandler(__FILE__,__LINE__,
// 			 "reading file '"+m_filename
// 			 +"': unknown element type
// '"+std::to_string(elementType)+"'", 			 ErrorHandler::normal);
//     }
//     __elementType[i] = elementType;
//     __references[i] = this->_getInteger(); // physical reference
//     this->_getInteger(); // drops entity reference

//     const int numberOfVertices = this->_getInteger();
//     if (numberOfVertices != __numberOfPrimitiveNodes[elementType]) {
//       std::stringstream errorMsg;
//       errorMsg << "reading file '" <<m_filename
// 	       << "':number of vertices (" << numberOfVertices
// 	       << ") does not match expected ("
// 	       << __numberOfPrimitiveNodes[elementType] << ")" << std::ends;

//       throw ErrorHandler(__FILE__,__LINE__,
// 			 errorMsg.str(),
// 			 ErrorHandler::normal);
//     }
//     __elementVertices[i].resize(numberOfVertices);
//     for (int j=0; j<numberOfVertices; ++j) {
//       __elementVertices[i][j] = this->_getInteger();
//     }
//   }
// }

void
GmshReader::__readElements2_2()
{
  const int numberOfElements = this->_getInteger();
  std::cout << "- Number Of Elements: " << numberOfElements << '\n';
  if (numberOfElements < 0) {
    throw ErrorHandler(__FILE__, __LINE__, "reading file '" + m_filename + "': number of elements is negative",
                       ErrorHandler::normal);
  }

  __elementNumber.resize(numberOfElements);
  __elementType.resize(numberOfElements);
  __references.resize(numberOfElements);
  __elementVertices.resize(numberOfElements);

  if (not __binary) {
    for (int i = 0; i < numberOfElements; ++i) {
      __elementNumber[i]    = this->_getInteger();
      const int elementType = this->_getInteger() - 1;

      if ((elementType < 0) or (elementType > 14)) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': unknown element type '" + std::to_string(elementType) +
                             "'",
                           ErrorHandler::normal);
      }
      __elementType[i]       = elementType;
      const int numberOfTags = this->_getInteger();
      __references[i]        = this->_getInteger();   // physical reference
      for (int tag = 1; tag < numberOfTags; ++tag) {
        this->_getInteger();   // drops remaining tags
      }

      const int numberOfVertices = __numberOfPrimitiveNodes[elementType];
      __elementVertices[i].resize(numberOfVertices);
      for (int j = 0; j < numberOfVertices; ++j) {
        __elementVertices[i][j] = this->_getInteger();
      }
    }
  } else {
    // fseek(__ifh,1L,SEEK_CUR);
    // int i=0;
    // for (;i<numberOfElements;) {
    //   int elementType = 0;
    //   fread(reinterpret_cast<char*>(&elementType),sizeof(int),1,__ifh);
    //   if (__convertEndian) {
    // 	invertEndianess(elementType);
    //   }
    //   --elementType;

    //   if ((elementType < 0) or (elementType > 14)) {
    // 	 throw ErrorHandler(__FILE__,__LINE__,
    // 	 		   "reading file '"+m_filename
    // 	 		   +"': unknown element type
    // '"+std::to_string(elementType)+"'",
    // ErrorHandler::normal);
    //   }

    //   int elementTypeNumber = 0;
    //   fread(reinterpret_cast<char*>(&elementTypeNumber),sizeof(int),1,__ifh);
    //   if (__convertEndian) {
    // 	invertEndianess(elementTypeNumber);
    //   }

    //   int numberOfTags = 0;
    //   fread(reinterpret_cast<char*>(&numberOfTags),sizeof(int),1,__ifh);
    //   if (__convertEndian) {
    // 	invertEndianess(numberOfTags);
    //   }

    //   for (int k=0; k<elementTypeNumber; ++k) {
    // 	__elementType[i] = elementType;

    // 	int elementNumber = 0;
    // 	fread(reinterpret_cast<char*>(&elementNumber),sizeof(int),1,__ifh);

    // 	int reference = 0;
    // 	fread(reinterpret_cast<char*>(&reference),sizeof(int),1,__ifh);

    // 	__references[i] = reference; // physical reference
    // 	for (int tag=1; tag<numberOfTags; ++tag) {
    // 	  int t;
    // 	  fread(reinterpret_cast<char*>(&t),sizeof(int),1,__ifh);
    // 	}

    // 	const int numberOfVertices = __numberOfPrimitiveNodes[elementType];
    // 	__elementVertices[i].resize(numberOfVertices);
    // 	fread(reinterpret_cast<char*>(&(__elementVertices[i][0])),sizeof(int),numberOfVertices,__ifh);
    // 	++i;
    //   }
    // }

    // if (__convertEndian) {
    //   for (size_t i=0; i<__references.size(); ++i) {
    // 	invertEndianess(__references[i]);
    //   }
    //   for (size_t i=0; i<__elementVertices.size(); ++i) {
    // 	for (size_t j=0; j<__elementVertices[i].size(); ++i) {
    // 	  invertEndianess(__elementVertices[i][j]);
    // 	}
    //   }
    // }
  }
}

void
GmshReader::__readPhysicalNames2_2()
{
  const int number_of_names = this->_getInteger();
  for (int i = 0; i < number_of_names; ++i) {
    const int physical_dimension = this->_getInteger();
    const int physical_number    = this->_getInteger();
    std::string physical_name;
    m_fin >> physical_name;
    physical_name = std::regex_replace(physical_name, std::regex("(\")"), "");

    PhysicalRefId physical_ref_id(physical_dimension, RefId(physical_number, physical_name));
    auto searched_physical_ref_id = m_physical_ref_map.find(physical_number);
    if (searched_physical_ref_id != m_physical_ref_map.end()) {
      std::cerr << "Physical reference id '" << physical_ref_id << "' already defined as '"
                << searched_physical_ref_id->second << "'!";
      std::exit(0);
    }
    m_physical_ref_map[physical_number] = physical_ref_id;
  }
}

template <size_t Dimension>
void
GmshReader::__computeCellFaceAndFaceNodeConnectivities(ConnectivityDescriptor& descriptor)
{
  static_assert((Dimension == 2) or (Dimension == 3), "Invalid dimension to compute cell-face connectivities");
  using CellFaceInfo = std::tuple<CellId, unsigned short, bool>;
  using Face         = ConnectivityFace<Dimension>;

  const auto& node_number_vector = descriptor.node_number_vector;
  Array<unsigned short> cell_nb_faces(descriptor.cell_to_node_vector.size());
  std::map<Face, std::vector<CellFaceInfo>> face_cells_map;
  for (CellId j = 0; j < descriptor.cell_to_node_vector.size(); ++j) {
    const auto& cell_nodes = descriptor.cell_to_node_vector[j];

    if constexpr (Dimension == 2) {
      switch (descriptor.cell_type_vector[j]) {
      case CellType::Triangle: {
        cell_nb_faces[j] = 3;
        // face 0
        Face f0({cell_nodes[1], cell_nodes[2]}, node_number_vector);
        face_cells_map[f0].emplace_back(std::make_tuple(j, 0, f0.reversed()));

        // face 1
        Face f1({cell_nodes[2], cell_nodes[0]}, node_number_vector);
        face_cells_map[f1].emplace_back(std::make_tuple(j, 1, f1.reversed()));

        // face 2
        Face f2({cell_nodes[0], cell_nodes[1]}, node_number_vector);
        face_cells_map[f2].emplace_back(std::make_tuple(j, 2, f2.reversed()));
        break;
      }
      case CellType::Quadrangle: {
        cell_nb_faces[j] = 4;
        // face 0
        Face f0({cell_nodes[0], cell_nodes[1]}, node_number_vector);
        face_cells_map[f0].emplace_back(std::make_tuple(j, 0, f0.reversed()));

        // face 1
        Face f1({cell_nodes[1], cell_nodes[2]}, node_number_vector);
        face_cells_map[f1].emplace_back(std::make_tuple(j, 1, f1.reversed()));

        // face 2
        Face f2({cell_nodes[2], cell_nodes[3]}, node_number_vector);
        face_cells_map[f2].emplace_back(std::make_tuple(j, 2, f2.reversed()));

        // face 3
        Face f3({cell_nodes[3], cell_nodes[0]}, node_number_vector);
        face_cells_map[f3].emplace_back(std::make_tuple(j, 3, f3.reversed()));
        break;
      }
      default: {
        std::cerr << name(descriptor.cell_type_vector[j]) << ": unexpected cell type in dimension 2!\n";
        std::terminate();
      }
      }
    } else if constexpr (Dimension == 3) {
      switch (descriptor.cell_type_vector[j]) {
      case CellType::Tetrahedron: {
        cell_nb_faces[j] = 4;
        // face 0
        Face f0({cell_nodes[1], cell_nodes[2], cell_nodes[3]}, node_number_vector);
        face_cells_map[f0].emplace_back(std::make_tuple(j, 0, f0.reversed()));

        // face 1
        Face f1({cell_nodes[0], cell_nodes[3], cell_nodes[2]}, node_number_vector);
        face_cells_map[f1].emplace_back(std::make_tuple(j, 1, f1.reversed()));

        // face 2
        Face f2({cell_nodes[0], cell_nodes[1], cell_nodes[3]}, node_number_vector);
        face_cells_map[f2].emplace_back(std::make_tuple(j, 2, f2.reversed()));

        // face 3
        Face f3({cell_nodes[0], cell_nodes[2], cell_nodes[1]}, node_number_vector);
        face_cells_map[f3].emplace_back(std::make_tuple(j, 3, f3.reversed()));
        break;
      }
      case CellType::Hexahedron: {
        // face 0
        Face f0({cell_nodes[3], cell_nodes[2], cell_nodes[1], cell_nodes[0]}, node_number_vector);
        face_cells_map[f0].emplace_back(std::make_tuple(j, 0, f0.reversed()));

        // face 1
        Face f1({cell_nodes[4], cell_nodes[5], cell_nodes[6], cell_nodes[7]}, node_number_vector);
        face_cells_map[f1].emplace_back(std::make_tuple(j, 1, f1.reversed()));

        // face 2
        Face f2({cell_nodes[0], cell_nodes[4], cell_nodes[7], cell_nodes[3]}, node_number_vector);
        face_cells_map[f2].emplace_back(std::make_tuple(j, 2, f2.reversed()));

        // face 3
        Face f3({cell_nodes[1], cell_nodes[2], cell_nodes[6], cell_nodes[5]}, node_number_vector);
        face_cells_map[f3].emplace_back(std::make_tuple(j, 3, f3.reversed()));

        // face 4
        Face f4({cell_nodes[0], cell_nodes[1], cell_nodes[5], cell_nodes[4]}, node_number_vector);
        face_cells_map[f4].emplace_back(std::make_tuple(j, 4, f4.reversed()));

        // face 5
        Face f5({cell_nodes[3], cell_nodes[7], cell_nodes[6], cell_nodes[2]}, node_number_vector);
        face_cells_map[f5].emplace_back(std::make_tuple(j, 5, f5.reversed()));

        cell_nb_faces[j] = 6;
        break;
      }
      default: {
        std::cerr << name(descriptor.cell_type_vector[j]) << ": unexpected cell type in dimension 3!\n";
        std::terminate();
      }
      }
    }
  }

  {
    descriptor.cell_to_face_vector.resize(descriptor.cell_to_node_vector.size());
    for (CellId j = 0; j < descriptor.cell_to_face_vector.size(); ++j) {
      descriptor.cell_to_face_vector[j].resize(cell_nb_faces[j]);
    }
    FaceId l = 0;
    for (const auto& face_cells_vector : face_cells_map) {
      const auto& cells_vector = face_cells_vector.second;
      for (unsigned short lj = 0; lj < cells_vector.size(); ++lj) {
        const auto& [cell_number, cell_local_face, reversed]         = cells_vector[lj];
        descriptor.cell_to_face_vector[cell_number][cell_local_face] = l;
      }
      ++l;
    }
  }

  {
    descriptor.cell_face_is_reversed_vector.resize(descriptor.cell_to_node_vector.size());
    for (CellId j = 0; j < descriptor.cell_face_is_reversed_vector.size(); ++j) {
      descriptor.cell_face_is_reversed_vector[j] = Array<bool>(cell_nb_faces[j]);
    }
    for (const auto& face_cells_vector : face_cells_map) {
      const auto& cells_vector = face_cells_vector.second;
      for (unsigned short lj = 0; lj < cells_vector.size(); ++lj) {
        const auto& [cell_number, cell_local_face, reversed]                  = cells_vector[lj];
        descriptor.cell_face_is_reversed_vector[cell_number][cell_local_face] = reversed;
      }
    }
  }

  {
    descriptor.face_to_node_vector.resize(face_cells_map.size());
    int l = 0;
    for (const auto& face_info : face_cells_map) {
      const Face& face                  = face_info.first;
      descriptor.face_to_node_vector[l] = face.nodeIdList();
      ++l;
    }
  }

  {
    // Face numbers may change if numbers are provided in the file
    descriptor.face_number_vector.resize(face_cells_map.size());
    for (size_t l = 0; l < face_cells_map.size(); ++l) {
      descriptor.face_number_vector[l] = l;
    }
  }
}

template <size_t Dimension>
void
GmshReader::__computeFaceEdgeAndEdgeNodeAndCellEdgeConnectivities(ConnectivityDescriptor& descriptor)
{
  static_assert(Dimension == 3, "Invalid dimension to compute face-edge connectivities");
  using FaceEdgeInfo = std::tuple<FaceId, unsigned short, bool>;
  using Edge         = ConnectivityFace<2>;

  const auto& node_number_vector = descriptor.node_number_vector;
  Array<unsigned short> face_nb_edges(descriptor.face_to_node_vector.size());
  std::map<Edge, std::vector<FaceEdgeInfo>> edge_faces_map;
  for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
    const auto& face_nodes = descriptor.face_to_node_vector[l];

    face_nb_edges[l] = face_nodes.size();
    for (size_t r = 0; r < face_nodes.size() - 1; ++r) {
      Edge e({face_nodes[r], face_nodes[r + 1]}, node_number_vector);
      edge_faces_map[e].emplace_back(std::make_tuple(l, r, e.reversed()));
    }
    {
      Edge e({face_nodes[face_nodes.size() - 1], face_nodes[0]}, node_number_vector);
      edge_faces_map[e].emplace_back(std::make_tuple(l, face_nodes.size() - 1, e.reversed()));
    }
  }

  std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_id_map;
  {
    descriptor.face_to_edge_vector.resize(descriptor.face_to_node_vector.size());
    for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
      descriptor.face_to_edge_vector[l].resize(face_nb_edges[l]);
    }
    EdgeId e = 0;
    for (const auto& edge_faces_vector : edge_faces_map) {
      const auto& faces_vector = edge_faces_vector.second;
      for (unsigned short l = 0; l < faces_vector.size(); ++l) {
        const auto& [face_number, face_local_edge, reversed]         = faces_vector[l];
        descriptor.face_to_edge_vector[face_number][face_local_edge] = e;
      }
      edge_id_map[edge_faces_vector.first] = e;
      ++e;
    }
  }

  {
    descriptor.face_edge_is_reversed_vector.resize(descriptor.face_to_node_vector.size());
    for (FaceId j = 0; j < descriptor.face_edge_is_reversed_vector.size(); ++j) {
      descriptor.face_edge_is_reversed_vector[j] = Array<bool>(face_nb_edges[j]);
    }
    for (const auto& edge_faces_vector : edge_faces_map) {
      const auto& faces_vector = edge_faces_vector.second;
      for (unsigned short lj = 0; lj < faces_vector.size(); ++lj) {
        const auto& [face_number, face_local_edge, reversed]                  = faces_vector[lj];
        descriptor.face_edge_is_reversed_vector[face_number][face_local_edge] = reversed;
      }
    }
  }

  {
    descriptor.edge_to_node_vector.resize(edge_faces_map.size());
    int e = 0;
    for (const auto& edge_info : edge_faces_map) {
      const Edge& edge                  = edge_info.first;
      descriptor.edge_to_node_vector[e] = edge.nodeIdList();
      ++e;
    }
  }

  {
    // Edge numbers may change if numbers are provided in the file
    descriptor.edge_number_vector.resize(edge_faces_map.size());
    for (size_t e = 0; e < edge_faces_map.size(); ++e) {
      descriptor.edge_number_vector[e] = e;
    }
  }

  {
    descriptor.cell_to_node_vector.reserve(descriptor.cell_to_node_vector.size());
    for (CellId j = 0; j < descriptor.cell_to_node_vector.size(); ++j) {
      const auto& cell_nodes = descriptor.cell_to_node_vector[j];

      switch (descriptor.cell_type_vector[j]) {
      case CellType::Tetrahedron: {
        constexpr int local_edge[6][2] = {{0, 1}, {0, 2}, {0, 3}, {1, 2}, {2, 3}, {3, 1}};
        std::vector<unsigned int> cell_edge_vector;
        cell_edge_vector.reserve(6);
        for (int i_edge = 0; i_edge < 6; ++i_edge) {
          const auto e = local_edge[i_edge];
          Edge edge{{cell_nodes[e[0]], cell_nodes[e[1]]}, node_number_vector};
          auto i = edge_id_map.find(edge);
          if (i == edge_id_map.end()) {
            std::cerr << "could not find this edge!\n";
            std::terminate();
          }
          cell_edge_vector.push_back(i->second);
        }
        descriptor.cell_to_edge_vector.emplace_back(cell_edge_vector);
        break;
      }
      case CellType::Hexahedron: {
        constexpr int local_edge[12][2] = {{0, 1}, {1, 2}, {2, 3}, {3, 0}, {4, 5}, {5, 6},
                                           {6, 7}, {7, 4}, {0, 4}, {1, 5}, {2, 6}, {3, 7}};
        std::vector<unsigned int> cell_edge_vector;
        cell_edge_vector.reserve(12);
        for (int i_edge = 0; i_edge < 12; ++i_edge) {
          const auto e = local_edge[i_edge];
          Edge edge{{cell_nodes[e[0]], cell_nodes[e[1]]}, node_number_vector};
          auto i = edge_id_map.find(edge);
          if (i == edge_id_map.end()) {
            std::cerr << "could not find this edge!\n";
            std::terminate();
          }
          cell_edge_vector.push_back(i->second);
        }
        descriptor.cell_to_edge_vector.emplace_back(cell_edge_vector);
        break;
      }
      default: {
        std::cerr << name(descriptor.cell_type_vector[j]) << ": unexpected cell type in dimension 3!\n";
        std::terminate();
      }
      }
    }
  }
}

void
GmshReader::__proceedData()
{
  TinyVector<15, int> elementNumber = zero;
  std::vector<std::set<size_t>> elementReferences(15);

  for (size_t i = 0; i < __elementType.size(); ++i) {
    const int elementType = __elementType[i];
    elementNumber[elementType]++;
    elementReferences[elementType].insert(__references[i]);
  }

  for (size_t i = 0; i < elementNumber.dimension(); ++i) {
    if (elementNumber[i] > 0) {
      std::cout << "  - Number of " << __primitivesNames[i] << ": " << elementNumber[i];
      if (not(__supportedPrimitives[i])) {
        std::cout << " [" << rang::fg::yellow << "IGNORED" << rang::style::reset << "]";
      } else {
        std::cout << " references: ";
        for (std::set<size_t>::const_iterator iref = elementReferences[i].begin(); iref != elementReferences[i].end();
             ++iref) {
          if (iref != elementReferences[i].begin())
            std::cout << ',';
          std::cout << *iref;
        }

        switch (i) {
        // Supported entities
        case 0: {   // edges
          __edges = Array<Edge>(elementNumber[i]);
          __edges_ref.resize(elementNumber[i]);
          __edges_number.resize(elementNumber[i]);
          break;
        }
        case 1: {   // triangles
          __triangles = Array<Triangle>(elementNumber[i]);
          __triangles_ref.resize(elementNumber[i]);
          __triangles_number.resize(elementNumber[i]);
          break;
        }
        case 2: {   // quadrangles
          __quadrangles = Array<Quadrangle>(elementNumber[i]);
          __quadrangles_ref.resize(elementNumber[i]);
          __quadrangles_number.resize(elementNumber[i]);
          break;
        }
        case 3: {   // tetrahedra
          __tetrahedra = Array<Tetrahedron>(elementNumber[i]);
          __tetrahedra_ref.resize(elementNumber[i]);
          __tetrahedra_number.resize(elementNumber[i]);
          break;
        }
        case 4: {   // hexahedra
          __hexahedra = Array<Hexahedron>(elementNumber[i]);
          __hexahedra_ref.resize(elementNumber[i]);
          __hexahedra_number.resize(elementNumber[i]);
          break;
        }
        case 14: {   // point
          __points = Array<Point>(elementNumber[i]);
          __points_ref.resize(elementNumber[i]);
          __points_number.resize(elementNumber[i]);
          break;
        }
          // Unsupported entities
        case 5:    // prism
        case 6:    // pyramid
        case 7:    // second order edge
        case 8:    // second order triangle
        case 9:    // second order quadrangle
        case 10:   // second order tetrahedron
        case 11:   // second order hexahedron
        case 12:   // second order prism
        case 13:   // second order pyramid
        default: {
          throw ErrorHandler(__FILE__, __LINE__,
                             "reading file '" + m_filename + "': ff3d cannot read this kind of element",
                             ErrorHandler::normal);
        }
        }
      }
      std::cout << '\n';
    }
  }

  std::cout << "- Building correspondance table\n";
  int minNumber = std::numeric_limits<int>::max();
  int maxNumber = std::numeric_limits<int>::min();
  for (size_t i = 0; i < __verticesNumbers.size(); ++i) {
    const int& vertexNumber = __verticesNumbers[i];
    minNumber               = std::min(minNumber, vertexNumber);
    maxNumber               = std::max(maxNumber, vertexNumber);
  }

  if (minNumber < 0) {
    throw ErrorHandler(__FILE__, __LINE__,
                       "reading file '" + m_filename +
                         "': ff3d does not implement negative vertices "
                         "number",
                       ErrorHandler::normal);
  }

  // A value of -1 means that the vertex is unknown
  __verticesCorrepondance.resize(maxNumber + 1, -1);

  for (size_t i = 0; i < __verticesNumbers.size(); ++i) {
    __verticesCorrepondance[__verticesNumbers[i]] = i;
  }

  // reset element number to count them while filling structures
  elementNumber = zero;

  // Element structures filling
  for (size_t i = 0; i < __elementType.size(); ++i) {
    std::vector<int>& elementVertices = __elementVertices[i];
    switch (__elementType[i]) {
    // Supported entities
    case 0: {   // edge
      int& edgeNumber = elementNumber[0];
      const int a     = __verticesCorrepondance[elementVertices[0]];
      const int b     = __verticesCorrepondance[elementVertices[1]];
      if ((a < 0) or (b < 0)) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                             " [bad vertices definition]",
                           ErrorHandler::normal);
      }
      __edges[edgeNumber]        = Edge(a, b);
      __edges_ref[edgeNumber]    = __references[i];
      __edges_number[edgeNumber] = __elementNumber[i];
      edgeNumber++;   // one more edge
      break;
    }
    case 1: {   // triangles
      int& triangleNumber = elementNumber[1];

      const int a = __verticesCorrepondance[elementVertices[0]];
      const int b = __verticesCorrepondance[elementVertices[1]];
      const int c = __verticesCorrepondance[elementVertices[2]];
      if ((a < 0) or (b < 0) or (c < 0)) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                             " [bad vertices definition]",
                           ErrorHandler::normal);
      }
      __triangles[triangleNumber]        = Triangle(a, b, c);
      __triangles_ref[triangleNumber]    = __references[i];
      __triangles_number[triangleNumber] = __elementNumber[i];
      triangleNumber++;   // one more triangle
      break;
    }
    case 2: {   // quadrangle
      int& quadrilateralNumber = elementNumber[2];

      const int a = __verticesCorrepondance[elementVertices[0]];
      const int b = __verticesCorrepondance[elementVertices[1]];
      const int c = __verticesCorrepondance[elementVertices[2]];
      const int d = __verticesCorrepondance[elementVertices[3]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0)) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                             " [bad vertices definition]",
                           ErrorHandler::normal);
      }
      __quadrangles[quadrilateralNumber]        = Quadrangle(a, b, c, d);
      __quadrangles_ref[quadrilateralNumber]    = __references[i];
      __quadrangles_number[quadrilateralNumber] = __elementNumber[i];
      quadrilateralNumber++;   // one more quadrangle
      break;
    }
    case 3: {   // tetrahedra
      int& tetrahedronNumber = elementNumber[3];

      const int a = __verticesCorrepondance[elementVertices[0]];
      const int b = __verticesCorrepondance[elementVertices[1]];
      const int c = __verticesCorrepondance[elementVertices[2]];
      const int d = __verticesCorrepondance[elementVertices[3]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0)) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                             " [bad vertices definition]",
                           ErrorHandler::normal);
      }
      __tetrahedra[tetrahedronNumber]        = Tetrahedron(a, b, c, d);
      __tetrahedra_ref[tetrahedronNumber]    = __references[i];
      __tetrahedra_number[tetrahedronNumber] = __elementNumber[i];
      tetrahedronNumber++;   // one more tetrahedron
      break;
    }
    case 4: {   // hexaredron
      int& hexahedronNumber = elementNumber[4];

      const int a = __verticesCorrepondance[elementVertices[0]];
      const int b = __verticesCorrepondance[elementVertices[1]];
      const int c = __verticesCorrepondance[elementVertices[2]];
      const int d = __verticesCorrepondance[elementVertices[3]];
      const int e = __verticesCorrepondance[elementVertices[4]];
      const int f = __verticesCorrepondance[elementVertices[5]];
      const int g = __verticesCorrepondance[elementVertices[6]];
      const int h = __verticesCorrepondance[elementVertices[7]];
      if ((a < 0) or (b < 0) or (c < 0) or (d < 0) or (e < 0) or (f < 0) or (g < 0) or (h < 0)) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': error reading element " + std::to_string(i) +
                             " [bad vertices definition]",
                           ErrorHandler::normal);
      }
      __hexahedra[hexahedronNumber]        = Hexahedron(a, b, c, d, e, f, g, h);
      __hexahedra_ref[hexahedronNumber]    = __references[i];
      __hexahedra_number[hexahedronNumber] = __elementNumber[i];
      hexahedronNumber++;   // one more hexahedron
      break;
    }
      // Unsupported entities
    case 14: {   // point
      int& point_number             = elementNumber[14];
      const int a                   = __verticesCorrepondance[elementVertices[0]];
      __points[point_number]        = a;
      __points_ref[point_number]    = __references[i];
      __points_number[point_number] = __elementNumber[i];
      point_number++;
      break;
    }
    case 5:      // prism
    case 6:      // pyramid
    case 7:      // second order edge
    case 8:      // second order triangle
    case 9:      // second order quadrangle
    case 10:     // second order tetrahedron
    case 11:     // second order hexahedron
    case 12:     // second order prism
    case 13: {   // second order pyramid
    }
    default: {
      throw ErrorHandler(__FILE__, __LINE__, "reading file '" + m_filename + "': ff3d cannot read this kind of element",
                         ErrorHandler::normal);
    }
    }
  }

  TinyVector<15, int> dimension0_mask = zero;
  dimension0_mask[14]                 = 1;
  TinyVector<15, int> dimension1_mask = zero;
  dimension1_mask[0]                  = 1;
  dimension1_mask[7]                  = 1;
  TinyVector<15, int> dimension2_mask = zero;
  dimension2_mask[1]                  = 1;
  dimension2_mask[2]                  = 1;
  dimension2_mask[8]                  = 1;
  dimension2_mask[9]                  = 1;
  TinyVector<15, int> dimension3_mask = zero;
  dimension3_mask[3]                  = 1;
  dimension3_mask[4]                  = 1;
  dimension3_mask[5]                  = 1;
  dimension3_mask[6]                  = 1;
  dimension3_mask[10]                 = 1;
  dimension3_mask[11]                 = 1;
  dimension3_mask[12]                 = 1;
  dimension3_mask[13]                 = 1;

  std::cout << "- dimension 0 entities: " << (dimension0_mask, elementNumber) << '\n';
  std::cout << "- dimension 1 entities: " << (dimension1_mask, elementNumber) << '\n';
  std::cout << "- dimension 2 entities: " << (dimension2_mask, elementNumber) << '\n';
  std::cout << "- dimension 3 entities: " << (dimension3_mask, elementNumber) << '\n';
  if ((dimension3_mask, elementNumber) > 0) {
    const size_t nb_cells = (dimension3_mask, elementNumber);

    ConnectivityDescriptor descriptor;

    descriptor.node_number_vector = __verticesNumbers;
    descriptor.cell_type_vector.resize(nb_cells);
    descriptor.cell_number_vector.resize(nb_cells);
    descriptor.cell_to_node_vector.resize(nb_cells);

    const size_t nb_tetrahedra = __tetrahedra.size();
    for (size_t j = 0; j < nb_tetrahedra; ++j) {
      descriptor.cell_to_node_vector[j].resize(4);
      for (int r = 0; r < 4; ++r) {
        descriptor.cell_to_node_vector[j][r] = __tetrahedra[j][r];
      }
      descriptor.cell_type_vector[j]   = CellType::Tetrahedron;
      descriptor.cell_number_vector[j] = __tetrahedra_number[j];
    }
    const size_t nb_hexahedra = __hexahedra.size();
    for (size_t j = 0; j < nb_hexahedra; ++j) {
      const size_t jh = nb_tetrahedra + j;
      descriptor.cell_to_node_vector[jh].resize(8);
      for (int r = 0; r < 8; ++r) {
        descriptor.cell_to_node_vector[jh][r] = __hexahedra[j][r];
      }
      descriptor.cell_type_vector[jh]   = CellType::Hexahedron;
      descriptor.cell_number_vector[jh] = __hexahedra_number[j];
    }

    std::map<unsigned int, std::vector<unsigned int>> ref_cells_map;
    for (unsigned int r = 0; r < __tetrahedra_ref.size(); ++r) {
      const unsigned int elem_number = __tetrahedra_ref[r];
      const unsigned int& ref        = __tetrahedra_ref[r];
      ref_cells_map[ref].push_back(elem_number);
    }

    for (unsigned int j = 0; j < __hexahedra_ref.size(); ++j) {
      const size_t elem_number = nb_tetrahedra + j;
      const unsigned int& ref  = __hexahedra_ref[j];
      ref_cells_map[ref].push_back(elem_number);
    }

    for (const auto& ref_cell_list : ref_cells_map) {
      Array<CellId> cell_list(ref_cell_list.second.size());
      for (size_t j = 0; j < ref_cell_list.second.size(); ++j) {
        cell_list[j] = ref_cell_list.second[j];
      }
      const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_cell_list.first);
      descriptor.addRefItemList(RefCellList(physical_ref_id.refId(), cell_list));
    }

    this->__computeCellFaceAndFaceNodeConnectivities<3>(descriptor);

    const auto& node_number_vector = descriptor.node_number_vector;

    {
      using Face                                                                 = ConnectivityFace<3>;
      const std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map = [&] {
        std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map;
        for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
          const auto& node_vector                               = descriptor.face_to_node_vector[l];
          face_to_id_map[Face(node_vector, node_number_vector)] = l;
        }
        return face_to_id_map;
      }();

      std::unordered_map<int, FaceId> face_number_id_map = [&] {
        std::unordered_map<int, FaceId> face_number_id_map;
        for (size_t l = 0; l < descriptor.face_number_vector.size(); ++l) {
          face_number_id_map[descriptor.face_number_vector[l]] = l;
        }
        Assert(face_number_id_map.size() == descriptor.face_number_vector.size());
        return face_number_id_map;
      }();

      std::map<unsigned int, std::vector<unsigned int>> ref_faces_map;
      for (unsigned int f = 0; f < __triangles.size(); ++f) {
        const unsigned int face_id = [&] {
          auto i =
            face_to_id_map.find(Face({__triangles[f][0], __triangles[f][1], __triangles[f][2]}, node_number_vector));
          if (i == face_to_id_map.end()) {
            std::cerr << "face not found!\n";
            std::terminate();
          }
          return i->second;
        }();

        const unsigned int& ref = __triangles_ref[f];
        ref_faces_map[ref].push_back(face_id);

        if (descriptor.face_number_vector[face_id] != __quadrangles_number[f]) {
          if (auto i_face = face_number_id_map.find(__quadrangles_number[f]); i_face != face_number_id_map.end()) {
            const int other_face_id = i_face->second;
            std::swap(descriptor.face_number_vector[face_id], descriptor.face_number_vector[other_face_id]);

            face_number_id_map.erase(descriptor.face_number_vector[face_id]);
            face_number_id_map.erase(descriptor.face_number_vector[other_face_id]);

            face_number_id_map[descriptor.face_number_vector[face_id]]       = face_id;
            face_number_id_map[descriptor.face_number_vector[other_face_id]] = other_face_id;
          } else {
            face_number_id_map.erase(descriptor.face_number_vector[face_id]);
            descriptor.face_number_vector[face_id]                     = __quadrangles_number[f];
            face_number_id_map[descriptor.face_number_vector[face_id]] = face_id;
          }
        }
      }

      for (unsigned int f = 0; f < __quadrangles.size(); ++f) {
        const unsigned int face_id = [&] {
          auto i = face_to_id_map.find(
            Face({__quadrangles[f][0], __quadrangles[f][1], __quadrangles[f][2], __quadrangles[f][3]},
                 node_number_vector));
          if (i == face_to_id_map.end()) {
            std::cerr << "face not found!\n";
            std::terminate();
          }
          return i->second;
        }();

        const unsigned int& ref = __quadrangles_ref[f];
        ref_faces_map[ref].push_back(face_id);

        if (descriptor.face_number_vector[face_id] != __quadrangles_number[f]) {
          if (auto i_face = face_number_id_map.find(__quadrangles_number[f]); i_face != face_number_id_map.end()) {
            const int other_face_id = i_face->second;
            std::swap(descriptor.face_number_vector[face_id], descriptor.face_number_vector[other_face_id]);

            face_number_id_map.erase(descriptor.face_number_vector[face_id]);
            face_number_id_map.erase(descriptor.face_number_vector[other_face_id]);

            face_number_id_map[descriptor.face_number_vector[face_id]]       = face_id;
            face_number_id_map[descriptor.face_number_vector[other_face_id]] = other_face_id;
          } else {
            face_number_id_map.erase(descriptor.face_number_vector[face_id]);
            descriptor.face_number_vector[face_id]                     = __quadrangles_number[f];
            face_number_id_map[descriptor.face_number_vector[face_id]] = face_id;
          }
        }
      }

      for (const auto& ref_face_list : ref_faces_map) {
        Array<FaceId> face_list(ref_face_list.second.size());
        for (size_t j = 0; j < ref_face_list.second.size(); ++j) {
          face_list[j] = ref_face_list.second[j];
        }
        const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_face_list.first);
        descriptor.addRefItemList(RefFaceList{physical_ref_id.refId(), face_list});
      }
    }
    this->__computeFaceEdgeAndEdgeNodeAndCellEdgeConnectivities<3>(descriptor);

    {
      using Edge                                                                 = ConnectivityFace<2>;
      const auto& node_number_vector                                             = descriptor.node_number_vector;
      const std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map = [&] {
        std::unordered_map<Edge, EdgeId, typename Edge::Hash> edge_to_id_map;
        for (EdgeId l = 0; l < descriptor.edge_to_node_vector.size(); ++l) {
          const auto& node_vector                               = descriptor.edge_to_node_vector[l];
          edge_to_id_map[Edge(node_vector, node_number_vector)] = l;
        }
        return edge_to_id_map;
      }();

      std::unordered_map<int, EdgeId> edge_number_id_map = [&] {
        std::unordered_map<int, EdgeId> edge_number_id_map;
        for (size_t l = 0; l < descriptor.edge_number_vector.size(); ++l) {
          edge_number_id_map[descriptor.edge_number_vector[l]] = l;
        }
        Assert(edge_number_id_map.size() == descriptor.edge_number_vector.size());
        return edge_number_id_map;
      }();

      std::map<unsigned int, std::vector<unsigned int>> ref_edges_map;
      for (unsigned int e = 0; e < __edges.size(); ++e) {
        const unsigned int edge_id = [&] {
          auto i = edge_to_id_map.find(Edge({__edges[e][0], __edges[e][1]}, node_number_vector));
          if (i == edge_to_id_map.end()) {
            std::cerr << "edge " << __edges[e][0] << " not found!\n";
            std::terminate();
          }
          return i->second;
        }();
        const unsigned int& ref = __edges_ref[e];
        ref_edges_map[ref].push_back(edge_id);

        if (descriptor.edge_number_vector[edge_id] != __edges_number[e]) {
          if (auto i_edge = edge_number_id_map.find(__edges_number[e]); i_edge != edge_number_id_map.end()) {
            const int other_edge_id = i_edge->second;
            std::swap(descriptor.edge_number_vector[edge_id], descriptor.edge_number_vector[other_edge_id]);

            edge_number_id_map.erase(descriptor.edge_number_vector[edge_id]);
            edge_number_id_map.erase(descriptor.edge_number_vector[other_edge_id]);

            edge_number_id_map[descriptor.edge_number_vector[edge_id]]       = edge_id;
            edge_number_id_map[descriptor.edge_number_vector[other_edge_id]] = other_edge_id;
          } else {
            edge_number_id_map.erase(descriptor.edge_number_vector[edge_id]);
            descriptor.edge_number_vector[edge_id]                     = __edges_number[e];
            edge_number_id_map[descriptor.edge_number_vector[edge_id]] = edge_id;
          }
        }
      }

      for (const auto& ref_edge_list : ref_edges_map) {
        Array<EdgeId> edge_list(ref_edge_list.second.size());
        for (size_t j = 0; j < ref_edge_list.second.size(); ++j) {
          edge_list[j] = ref_edge_list.second[j];
        }
        const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_edge_list.first);
        descriptor.addRefItemList(RefEdgeList{physical_ref_id.refId(), edge_list});
      }
    }

    std::map<unsigned int, std::vector<unsigned int>> ref_points_map;
    for (unsigned int r = 0; r < __points.size(); ++r) {
      const unsigned int point_number = __points[r];
      const unsigned int& ref         = __points_ref[r];
      ref_points_map[ref].push_back(point_number);
    }

    for (const auto& ref_point_list : ref_points_map) {
      Array<NodeId> point_list(ref_point_list.second.size());
      for (size_t j = 0; j < ref_point_list.second.size(); ++j) {
        point_list[j] = ref_point_list.second[j];
      }
      const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_point_list.first);
      descriptor.addRefItemList(RefNodeList(physical_ref_id.refId(), point_list));
    }

    descriptor.cell_owner_vector.resize(nb_cells);
    std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

    descriptor.face_owner_vector.resize(descriptor.face_number_vector.size());
    std::fill(descriptor.face_owner_vector.begin(), descriptor.face_owner_vector.end(), parallel::rank());

    descriptor.edge_owner_vector.resize(descriptor.edge_number_vector.size());
    std::fill(descriptor.edge_owner_vector.begin(), descriptor.edge_owner_vector.end(), parallel::rank());

    descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
    std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

    std::shared_ptr p_connectivity = Connectivity3D::build(descriptor);
    Connectivity3D& connectivity   = *p_connectivity;

    using MeshType = Mesh<Connectivity3D>;
    using Rd       = TinyVector<3, double>;

    NodeValue<Rd> xr(connectivity);
    for (NodeId i = 0; i < __vertices.size(); ++i) {
      xr[i][0] = __vertices[i][0];
      xr[i][1] = __vertices[i][1];
      xr[i][2] = __vertices[i][2];
    }
    m_mesh = std::make_shared<MeshType>(p_connectivity, xr);

  } else if ((dimension2_mask, elementNumber) > 0) {
    const size_t nb_cells = (dimension2_mask, elementNumber);

    ConnectivityDescriptor descriptor;

    descriptor.node_number_vector = __verticesNumbers;
    descriptor.cell_type_vector.resize(nb_cells);
    descriptor.cell_number_vector.resize(nb_cells);
    descriptor.cell_to_node_vector.resize(nb_cells);

    const size_t nb_triangles = __triangles.size();
    for (size_t j = 0; j < nb_triangles; ++j) {
      descriptor.cell_to_node_vector[j].resize(3);
      for (int r = 0; r < 3; ++r) {
        descriptor.cell_to_node_vector[j][r] = __triangles[j][r];
      }
      descriptor.cell_type_vector[j]   = CellType::Triangle;
      descriptor.cell_number_vector[j] = __triangles_number[j];
    }

    const size_t nb_quadrangles = __quadrangles.size();
    for (size_t j = 0; j < nb_quadrangles; ++j) {
      const size_t jq = j + nb_triangles;
      descriptor.cell_to_node_vector[jq].resize(4);
      for (int r = 0; r < 4; ++r) {
        descriptor.cell_to_node_vector[jq][r] = __quadrangles[j][r];
      }
      descriptor.cell_type_vector[jq]   = CellType::Quadrangle;
      descriptor.cell_number_vector[jq] = __quadrangles_number[j];
    }

    std::map<unsigned int, std::vector<unsigned int>> ref_cells_map;
    for (unsigned int r = 0; r < __triangles_ref.size(); ++r) {
      const unsigned int elem_number = __triangles_ref[r];
      const unsigned int& ref        = __triangles_ref[r];
      ref_cells_map[ref].push_back(elem_number);
    }

    for (unsigned int j = 0; j < __quadrangles_ref.size(); ++j) {
      const size_t elem_number = nb_triangles + j;
      const unsigned int& ref  = __quadrangles_ref[j];
      ref_cells_map[ref].push_back(elem_number);
    }

    for (const auto& ref_cell_list : ref_cells_map) {
      Array<CellId> cell_list(ref_cell_list.second.size());
      for (size_t j = 0; j < ref_cell_list.second.size(); ++j) {
        cell_list[j] = ref_cell_list.second[j];
      }
      const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_cell_list.first);
      descriptor.addRefItemList(RefCellList(physical_ref_id.refId(), cell_list));
    }

    this->__computeCellFaceAndFaceNodeConnectivities<2>(descriptor);

    using Face                                                                 = ConnectivityFace<2>;
    const auto& node_number_vector                                             = descriptor.node_number_vector;
    const std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map = [&] {
      std::unordered_map<Face, FaceId, typename Face::Hash> face_to_id_map;
      for (FaceId l = 0; l < descriptor.face_to_node_vector.size(); ++l) {
        const auto& node_vector                               = descriptor.face_to_node_vector[l];
        face_to_id_map[Face(node_vector, node_number_vector)] = l;
      }
      return face_to_id_map;
    }();

    std::unordered_map<int, FaceId> face_number_id_map = [&] {
      std::unordered_map<int, FaceId> face_number_id_map;
      for (size_t l = 0; l < descriptor.face_number_vector.size(); ++l) {
        face_number_id_map[descriptor.face_number_vector[l]] = l;
      }
      Assert(face_number_id_map.size() == descriptor.face_number_vector.size());
      return face_number_id_map;
    }();

    std::map<unsigned int, std::vector<unsigned int>> ref_faces_map;
    for (unsigned int e = 0; e < __edges.size(); ++e) {
      const unsigned int edge_id = [&] {
        auto i = face_to_id_map.find(Face({__edges[e][0], __edges[e][1]}, node_number_vector));
        if (i == face_to_id_map.end()) {
          std::cerr << "face " << __edges[e][0] << " not found!\n";
          std::terminate();
        }
        return i->second;
      }();
      const unsigned int& ref = __edges_ref[e];
      ref_faces_map[ref].push_back(edge_id);

      if (descriptor.face_number_vector[edge_id] != __edges_number[e]) {
        if (auto i_face = face_number_id_map.find(__edges_number[e]); i_face != face_number_id_map.end()) {
          const int other_edge_id = i_face->second;
          std::swap(descriptor.face_number_vector[edge_id], descriptor.face_number_vector[other_edge_id]);

          face_number_id_map.erase(descriptor.face_number_vector[edge_id]);
          face_number_id_map.erase(descriptor.face_number_vector[other_edge_id]);

          face_number_id_map[descriptor.face_number_vector[edge_id]]       = edge_id;
          face_number_id_map[descriptor.face_number_vector[other_edge_id]] = other_edge_id;
        } else {
          face_number_id_map.erase(descriptor.face_number_vector[edge_id]);
          descriptor.face_number_vector[edge_id]                     = __edges_number[e];
          face_number_id_map[descriptor.face_number_vector[edge_id]] = edge_id;
        }
      }
    }

    for (const auto& ref_face_list : ref_faces_map) {
      Array<FaceId> face_list(ref_face_list.second.size());
      for (size_t j = 0; j < ref_face_list.second.size(); ++j) {
        face_list[j] = ref_face_list.second[j];
      }
      const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_face_list.first);
      descriptor.addRefItemList(RefFaceList{physical_ref_id.refId(), face_list});
    }

    std::map<unsigned int, std::vector<unsigned int>> ref_points_map;
    for (unsigned int r = 0; r < __points.size(); ++r) {
      const unsigned int point_number = __points[r];
      const unsigned int& ref         = __points_ref[r];
      ref_points_map[ref].push_back(point_number);
    }

    for (const auto& ref_point_list : ref_points_map) {
      Array<NodeId> point_list(ref_point_list.second.size());
      for (size_t j = 0; j < ref_point_list.second.size(); ++j) {
        point_list[j] = ref_point_list.second[j];
      }
      const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_point_list.first);
      descriptor.addRefItemList(RefNodeList(physical_ref_id.refId(), point_list));
    }

    descriptor.cell_owner_vector.resize(nb_cells);
    std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

    descriptor.face_owner_vector.resize(descriptor.face_number_vector.size());
    std::fill(descriptor.face_owner_vector.begin(), descriptor.face_owner_vector.end(), parallel::rank());

    descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
    std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

    std::shared_ptr p_connectivity = Connectivity2D::build(descriptor);
    Connectivity2D& connectivity   = *p_connectivity;

    using MeshType = Mesh<Connectivity2D>;
    using Rd       = TinyVector<2, double>;

    NodeValue<Rd> xr(connectivity);
    for (NodeId i = 0; i < __vertices.size(); ++i) {
      xr[i][0] = __vertices[i][0];
      xr[i][1] = __vertices[i][1];
    }
    m_mesh = std::make_shared<MeshType>(p_connectivity, xr);

  } else if ((dimension1_mask, elementNumber) > 0) {
    const size_t nb_cells = (dimension1_mask, elementNumber);

    ConnectivityDescriptor descriptor;

    descriptor.node_number_vector = __verticesNumbers;
    descriptor.cell_type_vector.resize(nb_cells);
    descriptor.cell_number_vector.resize(nb_cells);
    descriptor.cell_to_node_vector.resize(nb_cells);

    for (size_t j = 0; j < nb_cells; ++j) {
      descriptor.cell_to_node_vector[j].resize(2);
      for (int r = 0; r < 2; ++r) {
        descriptor.cell_to_node_vector[j][r] = __edges[j][r];
      }
      descriptor.cell_type_vector[j]   = CellType::Line;
      descriptor.cell_number_vector[j] = __edges_number[j];
    }

    std::map<unsigned int, std::vector<unsigned int>> ref_points_map;
    for (unsigned int r = 0; r < __points.size(); ++r) {
      const unsigned int point_number = __points[r];
      const unsigned int& ref         = __points_ref[r];
      ref_points_map[ref].push_back(point_number);
    }

    for (const auto& ref_point_list : ref_points_map) {
      Array<NodeId> point_list(ref_point_list.second.size());
      for (size_t j = 0; j < ref_point_list.second.size(); ++j) {
        point_list[j] = ref_point_list.second[j];
      }
      const PhysicalRefId& physical_ref_id = m_physical_ref_map.at(ref_point_list.first);
      descriptor.addRefItemList(RefNodeList(physical_ref_id.refId(), point_list));
    }

    descriptor.cell_owner_vector.resize(nb_cells);
    std::fill(descriptor.cell_owner_vector.begin(), descriptor.cell_owner_vector.end(), parallel::rank());

    descriptor.node_owner_vector.resize(descriptor.node_number_vector.size());
    std::fill(descriptor.node_owner_vector.begin(), descriptor.node_owner_vector.end(), parallel::rank());

    std::shared_ptr p_connectivity = Connectivity1D::build(descriptor);
    Connectivity1D& connectivity   = *p_connectivity;

    using MeshType = Mesh<Connectivity1D>;
    using Rd       = TinyVector<1, double>;

    NodeValue<Rd> xr(connectivity);
    for (NodeId i = 0; i < __vertices.size(); ++i) {
      xr[i][0] = __vertices[i][0];
    }

    m_mesh = std::make_shared<MeshType>(p_connectivity, xr);

  } else {
    std::cerr << "*** using a dimension 0 mesh is forbidden!\n";
    std::exit(0);
  }
}

GmshReader::Keyword
GmshReader::__nextKeyword()
{
  GmshReader::Keyword kw;

  std::string aKeyword;
  m_fin >> aKeyword;
  if (not m_fin) {
    kw.second = EndOfFile;
    return kw;
  }

  KeywordList::iterator i = __keywordList.find(aKeyword.c_str());

  if (i != __keywordList.end()) {
    kw.first  = (*i).first;
    kw.second = (*i).second;
    return kw;
  }

  throw ErrorHandler(__FILE__, __LINE__, "reading file '" + m_filename + "': unknown keyword '" + aKeyword + "'",
                     ErrorHandler::normal);

  kw.first  = aKeyword;
  kw.second = Unknown;
  return kw;
}

void
GmshReader::__readGmshFormat2_2()
{
  std::cout << "- Reading Gmsh format 2.2\n";
  GmshReader::Keyword kw = std::make_pair("", Unknown);
  while (kw.second != EndOfFile) {
    kw = this->__nextKeyword();
    switch (kw.second) {
    case NODES: {
      this->__readVertices();
      if (this->__nextKeyword().second != ENDNODES) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': expecting $EndNodes, '" + kw.first + "' was found",
                           ErrorHandler::normal);
      }
      break;
    }
    case ELEMENTS: {
      this->__readElements2_2();
      kw = this->__nextKeyword();
      if (kw.second != ENDELEMENTS) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': expecting $EndElements, '" + kw.first + "' was found",
                           ErrorHandler::normal);
      }
      break;
    }
    case PHYSICALNAMES: {
      this->__readPhysicalNames2_2();
      if (this->__nextKeyword().second != ENDPHYSICALNAMES) {
        throw ErrorHandler(__FILE__, __LINE__,
                           "reading file '" + m_filename + "': expecting $EndNodes, '" + kw.first + "' was found",
                           ErrorHandler::normal);
      }
      break;
    }

    case EndOfFile: {
      break;
    }
    default: {
      throw ErrorHandler(__FILE__, __LINE__, "reading file '" + m_filename + "': unexpected '" + kw.first + "'",
                         ErrorHandler::normal);
    }
    }
  }
}
