#include <analysis/PyramidGaussQuadrature.hpp>
#include <utils/Exceptions.hpp>

void
PyramidGaussQuadrature::_buildPointAndWeightLists(const size_t degree)
{
  using R3 = TinyVector<3>;

  struct Descriptor
  {
    int id;
    double weight;
    std::vector<double> lambda_list;
  };

  auto fill_quadrature_points = [](auto descriptor_list, auto& point_list, auto& weight_list) {
    Assert(point_list.size() == weight_list.size());

    size_t k = 0;
    for (size_t i = 0; i < descriptor_list.size(); ++i) {
      const auto [id, unit_weight, value_list] = descriptor_list[i];

      const double w = (4. / 3) * unit_weight;

      switch (id) {
      case 1: {
        Assert(value_list.size() == 1);
        const double z = value_list[0];

        point_list[k]  = R3{0, 0, z};
        weight_list[k] = w;
        ++k;
        break;
      }
      case 2: {
        Assert(value_list.size() == 2);
        const double a = value_list[0];
        const double z = value_list[1];

        point_list[k + 0] = R3{+a, 0, z};
        point_list[k + 1] = R3{-a, 0, z};
        point_list[k + 2] = R3{0, +a, z};
        point_list[k + 3] = R3{0, -a, z};

        for (size_t l = 0; l < 4; ++l) {
          weight_list[k + l] = w;
        }

        k += 4;
        break;
      }
      case 3: {
        Assert(value_list.size() == 2);
        const double a = value_list[0];
        const double z = value_list[1];

        point_list[k + 0] = R3{+a, +a, z};
        point_list[k + 1] = R3{+a, -a, z};
        point_list[k + 2] = R3{-a, +a, z};
        point_list[k + 3] = R3{-a, -a, z};

        for (size_t l = 0; l < 4; ++l) {
          weight_list[k + l] = w;
        }

        k += 4;
        break;
      }
      case 4: {
        Assert(value_list.size() == 3);
        const double a = value_list[0];
        const double b = value_list[1];
        const double z = value_list[2];

        point_list[k + 0] = R3{+a, +b, z};
        point_list[k + 1] = R3{+a, -b, z};
        point_list[k + 2] = R3{-a, +b, z};
        point_list[k + 3] = R3{-a, -b, z};
        point_list[k + 4] = R3{+b, +a, z};
        point_list[k + 5] = R3{-b, +a, z};
        point_list[k + 6] = R3{+b, -a, z};
        point_list[k + 7] = R3{-b, -a, z};

        for (size_t l = 0; l < 8; ++l) {
          weight_list[k + l] = w;
        }

        k += 8;
        break;
      }
        // LCOV_EXCL_START
      default: {
        throw UnexpectedError("invalid quadrature id");
      }
        // LCOV_EXCL_STOP
      }
    }
  };

  switch (degree) {
  case 0:
  case 1: {
    constexpr size_t nb_points = 1;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.000000000000000e+00, {+2.500000000000000e-01}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 2: {
    constexpr size_t nb_points = 5;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 2.798666789016337e-01, {+5.606322125356171e-01}},
       Descriptor{3, 1.800333302745916e-01, {+5.269974873671749e-01, +1.292784570090256e-01}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 3: {
    constexpr size_t nb_points = 6;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.534506474854593e-01, {+3.032132711145601e-02}},
       Descriptor{1, 2.613312220748051e-01, {+5.656071879789744e-01}},
       Descriptor{3, 1.463045326099339e-01, {+5.845963663947116e-01, +1.666666666666667e-01}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 4: {
    constexpr size_t nb_points = 10;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 2.068834025895523e-01, {+1.251369531087465e-01}},
       Descriptor{1, 1.137418831706419e-01, {+6.772327888861374e-01}},
       Descriptor{2, 1.063245878893255e-01, {+6.505815563982326e-01, +3.223841495782137e-01}},
       Descriptor{3, 6.351909067062594e-02, {+6.579669971216900e-01, +3.924828389881535e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 5: {
    constexpr size_t nb_points = 15;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 6.773442693037113e-02, {+7.307094695547904e-01}},
       Descriptor{1, 6.470893518150579e-02, {+6.197232858190588e-03}},
       Descriptor{1, 1.772715490151452e-01, {+2.684458095343137e-01}},
       Descriptor{2, 5.910777216655192e-02, {+7.534406130793294e-01, +1.250000000000000e-01}},
       Descriptor{3, 6.537546219121122e-02, {+4.171520024257513e-01, +4.218217110028595e-01}},
       Descriptor{3, 4.808803786048134e-02, {+6.740225164778704e-01, +6.579572180745927e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 6: {
    constexpr size_t nb_points = 23;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.023699419233705e-01, {+1.335312170632148e-01}},
       Descriptor{1, 2.544552509057920e-02, {+8.083918187874604e-01}},
       Descriptor{1, 1.074435834226933e-01, {+3.784035206635531e-01}},
       Descriptor{2, 3.715744178992644e-02, {+4.210459518278233e-01, +5.563577402280808e-01}},
       Descriptor{2, 3.663269740345384e-02, {+8.358409250652439e-01, +9.682668434012107e-02}},
       Descriptor{3, 7.134885171305939e-02, {+5.134178134130217e-01, +2.554780750374050e-01}},
       Descriptor{3, 8.659461394440064e-03, {+8.719795336426682e-01, +3.348911098405843e-02}},
       Descriptor{3, 3.738678508995950e-02, {+4.773315577677307e-01, +2.776222122928558e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 7: {
    constexpr size_t nb_points = 31;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.005139817749384e-01, {+3.936504852592841e-01}},
       Descriptor{1, 1.571901760701542e-02, {+8.386341427229903e-01}},
       Descriptor{1, 2.499658963028166e-02, {+1.985131073852604e-05}},
       Descriptor{2, 2.871093750000000e-02, {+6.172133998483676e-01, +3.333333333333333e-01}},
       Descriptor{2, 2.669175929300292e-02, {+8.640987597877147e-01, +6.666666666666667e-02}},
       Descriptor{3, 3.572750182264944e-02, {+5.248875603037457e-01, +2.904549108425410e-01}},
       Descriptor{3, 2.951904528668866e-02, {+2.541968221946381e-01, +6.054783556814159e-01}},
       Descriptor{3, 6.594160872648229e-02, {+3.540511188101694e-01, +1.293188463105600e-01}},
       Descriptor{3, 1.333274388639106e-02, {+6.142719454511971e-01, +1.008633926811357e-04}},
       Descriptor{3, 1.476900623172677e-02, {+8.028224862699490e-01, +8.012951317750569e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 8: {
    constexpr size_t nb_points = 47;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 5.595524252285600e-02, {+7.395194949759915e-02}},
       Descriptor{1, 6.694668391641566e-02, {+4.806418077857804e-01}},
       Descriptor{1, 4.752523069833954e-03, {+8.978770012649402e-01}},
       Descriptor{2, 1.488586682102477e-02, {+2.536785615782182e-01, +7.039949439220020e-01}},
       Descriptor{2, 2.455698624881831e-02, {+7.102737577690728e-01, +1.599976229101533e-01}},
       Descriptor{2, 1.608138988371910e-02, {+6.364336235983890e-01, +3.474084640816018e-01}},
       Descriptor{2, 1.442915622061931e-02, {+6.233792819622643e-01, +1.127682420195144e-02}},
       Descriptor{3, 2.488836268558412e-02, {+5.122596817100590e-01, +6.351022006373874e-02}},
       Descriptor{3, 3.016542061786949e-02, {+3.796590137942965e-01, +4.628422688700697e-01}},
       Descriptor{3, 1.825943823062004e-02, {+6.914008694052951e-01, +1.917713050993898e-01}},
       Descriptor{3, 4.052705114084869e-03, {+8.674219079854986e-01, +1.630913438364360e-02}},
       Descriptor{3, 5.109969513593878e-02, {+3.171232262910623e-01, +2.368198703013063e-01}},
       Descriptor{4, 9.833683332222407e-03, {+8.937479716183564e-01, +4.052832634656467e-01, +5.005997974535450e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 9: {
    constexpr size_t nb_points = 62;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 3.517982458221282e-03, {+9.033806033428579e-01}},
       Descriptor{1, 4.216520928063209e-02, {+5.649644435959712e-01}},
       Descriptor{2, 8.521043257455639e-03, {+2.199358452633800e-01, +7.485612463519468e-01}},
       Descriptor{2, 2.087072229798968e-02, {+2.558015573793007e-01, +1.346997662955544e-01}},
       Descriptor{2, 1.073969585101612e-02, {+8.739242228416501e-01, +5.467616134251216e-02}},
       Descriptor{2, 2.521321637389669e-02, {+6.206970648203557e-01, +1.894626948405005e-01}},
       Descriptor{2, 1.961258547003659e-02, {+4.808872023980156e-01, +2.248423582708249e-02}},
       Descriptor{2, 1.643323197880765e-02, {+5.476633755989256e-01, +4.134082258927628e-01}},
       Descriptor{3, 1.830589291063876e-02, {+3.171678212066174e-01, +5.507723363912053e-01}},
       Descriptor{3, 1.130841811263377e-02, {+5.811421463327766e-01, +3.191414755591298e-01}},
       Descriptor{3, 2.507245299443831e-02, {+5.230813443611307e-01, +8.935238781868592e-02}},
       Descriptor{3, 4.419409904347655e-03, {+8.727886385585986e-01, +6.151583405729108e-02}},
       Descriptor{3, 4.065272607298719e-02, {+2.744783026131462e-01, +3.264219851596609e-01}},
       Descriptor{4, 1.219793270502279e-02, {+4.578740192946059e-01, +7.687455968654898e-01, +1.755724204367565e-01}},
       Descriptor{4, 6.516970715496498e-03, {+5.294592086619503e-01, +8.651632950329239e-01, +1.570840570495663e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 10: {
    constexpr size_t nb_points = 80;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 2.226887246752692e-03, {+9.176665489678424e-01}},
       Descriptor{1, 1.676862119517167e-02, {+7.090093740468727e-01}},
       Descriptor{1, 5.218524659101940e-02, {+2.271396740104223e-01}},
       Descriptor{1, 2.912684113372536e-02, {+6.874011069480644e-02}},
       Descriptor{2, 2.061250827144200e-02, {+6.542017017611934e-01, +1.239254499739905e-01}},
       Descriptor{2, 1.001318317045432e-02, {+7.663392799174616e-01, +2.096299791450872e-02}},
       Descriptor{2, 3.805894719789539e-03, {+8.042105610181555e-01, +1.657731354676645e-01}},
       Descriptor{2, 1.603356460939568e-02, {+2.171975788362678e-01, +4.934052093395356e-01}},
       Descriptor{2, 4.210573985824022e-03, {+2.093734735103665e-01, +7.896976713483934e-01}},
       Descriptor{2, 8.129732309410973e-03, {+5.283070905328926e-01, +4.716928890485879e-01}},
       Descriptor{3, 1.345646795638387e-02, {+3.468149019049753e-01, +2.011394567580291e-02}},
       Descriptor{3, 1.016376299548378e-02, {+5.423477423323926e-01, +3.682210199892779e-01}},
       Descriptor{3, 2.981645762492043e-02, {+4.296576954455671e-01, +1.266159486770112e-01}},
       Descriptor{3, 8.657127834237414e-03, {+7.187141460172495e-01, +2.433819982878694e-02}},
       Descriptor{3, 9.531999164931240e-03, {+7.624400857874878e-01, +1.270144250256904e-01}},
       Descriptor{3, 7.139582674758052e-04, {+9.642735008631930e-01, +1.067692573406055e-02}},
       Descriptor{3, 1.519293397619959e-02, {+2.789738869874377e-01, +5.970429731303039e-01}},
       Descriptor{4, 1.008533639437561e-02, {+7.312452983523516e-01, +3.711506789000326e-01, +2.175616631549504e-01}},
       Descriptor{4, 5.859338229960730e-03, {+9.219480705797174e-01, +4.196652283339353e-01, +4.137857207280887e-02}},
       Descriptor{4, 2.134779341185569e-02, {+1.995426151915722e-01, +4.434009744724249e-01, +3.261536572399062e-01}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 11: {
    constexpr size_t nb_points = 103;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.286363653695535e-02, {+6.929563964750883e-01}},
       Descriptor{1, 2.595721225427735e-03, {+9.110152344627592e-01}},
       Descriptor{1, 8.176385700475526e-03, {+2.503292907254847e-01}},
       Descriptor{2, 5.119414398042891e-03, {+3.095914411121506e-01, +5.040798390564714e-02}},
       Descriptor{2, 6.846996666462350e-03, {+7.699392208102979e-01, +1.178348626907169e-02}},
       Descriptor{2, 2.021405278036776e-02, {+7.002703811744613e-01, +1.010394090255615e-01}},
       Descriptor{2, 1.240017518258633e-02, {+4.609503542907282e-01, +3.427275760580460e-01}},
       Descriptor{2, 1.050060859922616e-02, {+2.992893821695951e-01, +6.286989821677651e-01}},
       Descriptor{2, 1.207523991498537e-02, {+5.911494375715570e-01, +2.582284852308509e-01}},
       Descriptor{2, 7.945494399371682e-03, {+1.861308147704910e-02, +4.358344954832800e-01}},
       Descriptor{2, 8.979308019639792e-03, {+1.603429097546466e-01, +2.192831941429789e-01}},
       Descriptor{3, 4.328610077172669e-03, {+1.445075355444747e-01, +7.879743305346877e-01}},
       Descriptor{3, 9.591505938503333e-03, {+5.936945097967551e-01, +3.089295886910713e-01}},
       Descriptor{3, 8.458896075424903e-03, {+2.749014180295009e-01, +1.781185684646454e-02}},
       Descriptor{3, 2.093917618186587e-02, {+2.968007724724139e-01, +1.112703360898260e-01}},
       Descriptor{3, 1.007335618421271e-02, {+5.984926370656903e-01, +2.444716310077481e-02}},
       Descriptor{3, 1.474935950898293e-02, {+6.103141530732866e-01, +1.204187551930976e-01}},
       Descriptor{3, 4.054320200259096e-03, {+3.734502308185636e-01, +5.775253255524150e-01}},
       Descriptor{3, 1.768012056788626e-02, {+2.624599146327740e-01, +4.916980190881633e-01}},
       Descriptor{3, 2.349367280688882e-02, {+3.686063571979057e-01, +2.631132571277226e-01}},
       Descriptor{3, 3.003553459527201e-03, {+8.702629344236114e-01, +1.908681361607608e-02}},
       Descriptor{3, 4.330579718905655e-03, {+8.102574500242316e-01, +1.169261992676211e-01}},
       Descriptor{4, 7.882977643914898e-03, {+7.896765746747297e-01, +3.552721383504759e-01, +1.796809539017722e-01}},
       Descriptor{4, 5.094911855326565e-03, {+9.280503494413060e-01, +4.148421023475579e-01, +3.535559961728909e-02}},
       Descriptor{4, 6.675422227745319e-03, {+2.331378908359926e-01, +5.562877919132911e-01, +4.146440202422456e-01}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 12: {
    constexpr size_t nb_points = 127;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 2.715854620871203e-02, {+4.570070042407191e-01}},
       Descriptor{1, 4.353142424489953e-04, {+9.624848010480895e-01}},
       Descriptor{1, 1.391222140872406e-02, {+2.656531684537741e-02}},
       Descriptor{2, 3.872881466492529e-03, {+6.175240822879710e-01, +5.374720825773919e-03}},
       Descriptor{2, 2.179679715936184e-03, {+9.538341781347592e-01, +3.942123626649530e-02}},
       Descriptor{2, 1.267133485574918e-02, {+5.950300580138582e-01, +7.619184421748394e-02}},
       Descriptor{2, 8.734641630353807e-03, {+4.353935982023290e-01, +5.246187553362018e-01}},
       Descriptor{2, 2.891467316158506e-03, {+1.329135556120804e-01, +8.387199403114260e-01}},
       Descriptor{2, 3.366868485115227e-03, {+7.971436348244696e-01, +2.007518977789794e-01}},
       Descriptor{2, 1.476038031344119e-03, {+3.107829504939940e-01, +6.826546910531470e-01}},
       Descriptor{2, 1.998444773599242e-02, {+4.689689670336921e-01, +3.171338246752144e-01}},
       Descriptor{2, 6.189271706444372e-03, {+6.329314929065794e-02, +1.270962903467187e-01}},
       Descriptor{3, 8.200787054788633e-03, {+1.169412447366524e-01, +6.712417429492287e-01}},
       Descriptor{3, 5.096515464581623e-03, {+4.771085065521420e-01, +4.586280271864225e-01}},
       Descriptor{3, 1.739298850473944e-02, {+3.479673540088846e-01, +1.193889061370197e-01}},
       Descriptor{3, 1.948797782027694e-02, {+2.174809398989127e-01, +2.574178486652196e-01}},
       Descriptor{3, 5.388962434670257e-03, {+6.742620754102585e-01, +2.260502558938976e-02}},
       Descriptor{3, 6.769117314513475e-03, {+6.541531257781691e-01, +8.746307755558261e-02}},
       Descriptor{3, 4.675876093665083e-03, {+2.604386899595714e-01, +6.714016412797261e-01}},
       Descriptor{3, 1.838172037009153e-02, {+2.744210786837723e-01, +4.760425504887897e-01}},
       Descriptor{3, 1.407964506457295e-02, {+5.125902273710998e-01, +2.277795581554027e-01}},
       Descriptor{3, 2.833700634607033e-03, {+8.687887353904908e-01, +5.850272176027390e-02}},
       Descriptor{3, 4.791372228177735e-03, {+7.118168180030836e-01, +2.205399573194206e-01}},
       Descriptor{3, 9.906806338191463e-03, {+3.675456953238826e-01, +2.548724290196442e-02}},
       Descriptor{4, 5.355743630233014e-03, {+8.489693748245684e-01, +5.120364631720311e-01, +1.114583653780480e-01}},
       Descriptor{4, 8.168185107182221e-04, {+9.738540313599703e-01, +7.467034094055778e-01, +3.392208246799127e-03}},
       Descriptor{4, 8.541753721892767e-03, {+3.230240551801264e-01, +6.255561325380413e-01, +3.272515164405167e-01}},
       Descriptor{4, 1.018769584471249e-02, {+1.945007634528372e-01, +6.980595936427326e-01, +1.480516686189832e-01}},
       Descriptor{4, 5.723677926726618e-03, {+3.289488538718914e-01, +8.538315289662523e-01, +2.487563023274973e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 13: {
    constexpr size_t nb_points = 152;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.203669565577547e-03, {+9.297400586425600e-01}},
       Descriptor{1, 1.890734566258640e-02, {+5.838703604447901e-01}},
       Descriptor{1, 2.412883548353303e-02, {+1.094245815067315e-01}},
       Descriptor{1, 1.430958560741041e-02, {+3.004359812964663e-01}},
       Descriptor{2, 1.462233467606358e-02, {+4.122389208832898e-01, +4.035176514416959e-01}},
       Descriptor{2, 1.310471335067461e-03, {+9.448240879795888e-01, +5.541396470307911e-03}},
       Descriptor{2, 1.315348594597115e-03, {+1.501230655029512e-01, +8.497929440328683e-01}},
       Descriptor{2, 6.041729387793333e-03, {+3.633828357756222e-01, +6.071950436217343e-01}},
       Descriptor{2, 8.289343691958597e-03, {+3.289712237345070e-01, +2.033792783786767e-02}},
       Descriptor{2, 1.293341373858936e-02, {+5.570493589160862e-01, +1.060119987373025e-01}},
       Descriptor{2, 5.638854191543832e-03, {+7.106035443742125e-01, +2.491041991532026e-01}},
       Descriptor{2, 4.504991398813604e-03, {+8.853174870868126e-01, +8.107956240970675e-02}},
       Descriptor{3, 3.636184206108946e-03, {+4.888475450384162e-01, +9.508151681060791e-03}},
       Descriptor{3, 3.888421168021828e-03, {+4.503177245667275e-01, +4.881936050921409e-01}},
       Descriptor{3, 4.023653208263601e-03, {+6.587295691554786e-01, +2.769083115660864e-01}},
       Descriptor{3, 9.559019492448712e-03, {+6.258827191179884e-01, +1.224809382042812e-01}},
       Descriptor{3, 1.938656716858041e-02, {+2.517896003319362e-01, +2.232305727986451e-01}},
       Descriptor{3, 9.687548837381962e-04, {+9.133819128044913e-01, +1.207165002118426e-02}},
       Descriptor{3, 5.022143502861996e-03, {+1.111786164172054e-01, +7.551872471669051e-01}},
       Descriptor{3, 1.448769412031482e-02, {+3.781492498737534e-01, +8.424825204838618e-02}},
       Descriptor{3, 4.599764313830424e-03, {+2.314619569157776e-01, +6.318857723177680e-01}},
       Descriptor{3, 1.107636061349879e-02, {+1.859866728257666e-01, +3.989297765923411e-01}},
       Descriptor{3, 4.692602004327276e-03, {+7.294149054289296e-01, +2.502635713295603e-02}},
       Descriptor{3, 2.851949824498801e-03, {+8.041924177792682e-01, +1.203800076513554e-01}},
       Descriptor{3, 1.323033835935081e-02, {+4.703608283841747e-01, +2.927035935703978e-01}},
       Descriptor{3, 1.119723439728825e-03, {+2.753565752001155e-01, +7.074274892155881e-01}},
       Descriptor{3, 9.182464621381196e-03, {+2.584044366257879e-01, +5.283274650941000e-01}},
       Descriptor{4, 4.564906537010502e-03, {+7.745630168641912e-01, +4.397893897194206e-01, +1.953840901290468e-01}},
       Descriptor{4, 5.475034834428800e-03, {+7.201414655134084e-01, +2.217479044308266e-01, +2.260035188498128e-02}},
       Descriptor{4, 9.922612076457428e-03, {+1.750984955638136e-01, +5.856079751486080e-01, +2.160852053363709e-01}},
       Descriptor{4, 1.103829121564573e-03, {+9.376254731805550e-01, +7.398267532513132e-01, +6.232437306564607e-02}},
       Descriptor{4, 2.362282939586705e-03, {+9.258882242137035e-01, +5.000209314552113e-01, +1.929314718254384e-02}},
       Descriptor{4, 6.306665849348717e-03, {+2.682073921041632e-01, +5.618519868919964e-01, +4.010246655048774e-01}},
       Descriptor{4, 6.754925131024099e-03, {+3.620795943973790e-01, +7.924562807098640e-01, +9.200529105614789e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 14: {
    constexpr size_t nb_points = 184;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 6.009828410914801e-04, {+9.455096078380157e-01}},
       Descriptor{1, 1.255487583666288e-02, {+6.138243564839928e-01}},
       Descriptor{1, 2.319377258130698e-02, {+2.454049718059621e-01}},
       Descriptor{1, 1.905556551345681e-02, {+4.183282539085985e-01}},
       Descriptor{2, 1.191795956205729e-02, {+4.218261367348597e-01, +4.017536316056771e-01}},
       Descriptor{2, 3.867184412332806e-03, {+8.465613435886886e-02, +8.958821481037450e-02}},
       Descriptor{2, 1.923763095174742e-03, {+9.516713196592316e-01, +3.128290548598175e-02}},
       Descriptor{2, 1.234179804493060e-03, {+1.221288178114563e-01, +8.656263522240111e-01}},
       Descriptor{2, 4.716648039115899e-03, {+3.459755786873036e-01, +5.615491563602975e-01}},
       Descriptor{2, 5.134044973384534e-03, {+3.250020921247040e-01, +1.398939848661482e-02}},
       Descriptor{2, 1.154564746408180e-03, {+8.527465617518474e-01, +1.655718296284047e-03}},
       Descriptor{2, 1.181264118995304e-02, {+6.382775647715264e-01, +6.493816225859560e-02}},
       Descriptor{2, 2.436937834631011e-03, {+6.723114741458501e-01, +3.226524499585241e-01}},
       Descriptor{2, 2.442180393951459e-03, {+8.537496059510902e-01, +1.428460235322913e-01}},
       Descriptor{2, 2.760272065188771e-03, {+3.235182852814910e-01, +6.650982181901243e-01}},
       Descriptor{2, 1.491875937488001e-02, {+4.969751483237657e-01, +1.927846766563645e-01}},
       Descriptor{3, 8.607885480520022e-03, {+5.041706295584495e-01, +5.324772752904031e-02}},
       Descriptor{3, 9.201054026224215e-04, {+6.385320288130391e-01, +3.499922479913266e-01}},
       Descriptor{3, 3.506580713358855e-03, {+7.643224482486065e-01, +1.609611231580418e-01}},
       Descriptor{3, 1.184517921997258e-02, {+5.613330619383329e-01, +1.603998892409423e-01}},
       Descriptor{3, 1.517200950119656e-02, {+2.878680539515507e-01, +1.373276390573706e-01}},
       Descriptor{3, 7.988570964665479e-04, {+9.016308382757633e-01, +1.032358209500777e-02}},
       Descriptor{3, 3.465322056721376e-03, {+9.459747575878658e-02, +7.677917941640774e-01}},
       Descriptor{3, 5.649017528916505e-03, {+2.627574405731676e-01, +4.943554173327801e-02}},
       Descriptor{3, 3.144992576543379e-03, {+3.864091536381419e-01, +5.560266141216673e-01}},
       Descriptor{3, 1.042467052306092e-02, {+2.282300577985542e-01, +4.962791997555249e-01}},
       Descriptor{3, 1.353353344199375e-03, {+7.057872269283543e-01, +2.008807154101613e-03}},
       Descriptor{3, 6.863946049178268e-04, {+9.147875524427646e-01, +4.956231677804342e-02}},
       Descriptor{3, 8.488466463152199e-03, {+5.088306451368273e-01, +3.463558696873174e-01}},
       Descriptor{3, 1.389764713997917e-03, {+2.183390551054974e-01, +7.454288363105724e-01}},
       Descriptor{3, 5.646871721661347e-03, {+2.052421004307298e-01, +6.415316471289261e-01}},
       Descriptor{3, 1.822286920356053e-02, {+2.910901390165372e-01, +3.062273833527194e-01}},
       Descriptor{3, 4.755535844463758e-03, {+7.486540248046429e-01, +4.936880562940795e-02}},
       Descriptor{4, 3.308393786732543e-03, {+7.378246466058306e-01, +4.928495648091606e-01, +2.369940069990296e-01}},
       Descriptor{4, 4.070045103160022e-03, {+8.589060620098202e-01, +3.738196822630493e-01, +2.776225858911670e-02}},
       Descriptor{4, 8.268489304502566e-03, {+2.243615064922758e-01, +6.317949456704877e-01, +2.592830550287851e-01}},
       Descriptor{4, 2.221378811027735e-03, {+9.001757758936474e-01, +6.261867885017652e-01, +7.806621986847780e-02}},
       Descriptor{4, 5.341491722009947e-04, {+9.789501849840820e-01, +6.256578399323836e-01, +4.430014519257858e-03}},
       Descriptor{4, 4.879462347378159e-03, {+2.560758373363789e-01, +5.119481066576588e-01, +4.542857276885983e-01}},
       Descriptor{4, 7.897297568864630e-03, {+2.974176206871381e-01, +7.755395865083232e-01, +1.176110467333258e-01}},
       Descriptor{4, 2.696628566117115e-03, {+2.907965956609688e-01, +6.192615227882655e-01, +7.834837395674860e-03}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 15: {
    constexpr size_t nb_points = 234;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 4.253301554860054e-04, {+9.516784060639266e-01}},
       Descriptor{1, 1.525748792328103e-02, {+2.989009214424757e-01}},
       Descriptor{2, 5.280176625857056e-03, {+7.001403574239718e-01, +1.037252232620472e-01}},
       Descriptor{2, 2.446394589904036e-03, {+9.193631484833984e-01, +6.052669044962753e-02}},
       Descriptor{2, 9.737060312972353e-04, {+1.101334889442361e-01, +8.776707213450818e-01}},
       Descriptor{2, 7.776124953823158e-03, {+4.465008567015498e-01, +3.306143190884451e-01}},
       Descriptor{2, 2.676970927679192e-03, {+8.020538575801487e-01, +1.891305716440682e-01}},
       Descriptor{2, 6.190812842666219e-03, {+5.301512452732464e-01, +1.554979319555147e-01}},
       Descriptor{2, 6.561865616728806e-04, {+9.732273442892753e-01, +4.905110329399738e-03}},
       Descriptor{2, 6.569940566182691e-03, {+5.055718663406419e-01, +6.238803484862593e-02}},
       Descriptor{2, 2.673087432145158e-03, {+4.618938260051262e-02, +5.328414641791853e-01}},
       Descriptor{2, 1.759495302160485e-03, {+6.230136406382800e-01, +3.769559380708412e-01}},
       Descriptor{2, 3.312857608561831e-03, {+2.950931045827234e-01, +6.814825140552475e-01}},
       Descriptor{2, 7.510345065720940e-03, {+4.776005857611240e-01, +2.309753566333424e-01}},
       Descriptor{2, 8.931665290893454e-03, {+3.689801580562112e-01, +5.033566690504305e-01}},
       Descriptor{3, 5.568418259966719e-03, {+3.723782904830595e-01, +4.235367529321636e-01}},
       Descriptor{3, 2.649457392781329e-03, {+5.341902002288372e-01, +4.129667979379346e-01}},
       Descriptor{3, 1.970289990554734e-03, {+6.063808134784291e-02, +7.864057957899899e-01}},
       Descriptor{3, 5.134839039660163e-03, {+4.777068770658753e-01, +3.265377521300765e-01}},
       Descriptor{3, 7.421925481559295e-03, {+1.987543120573730e-01, +5.291423306214975e-02}},
       Descriptor{3, 1.644857216298594e-03, {+7.599695611037520e-01, +6.930996189193400e-03}},
       Descriptor{3, 5.534536939100995e-03, {+5.055458420266429e-01, +1.799129130112892e-02}},
       Descriptor{3, 3.256281564205464e-03, {+2.206482423626575e-01, +8.304773708384203e-03}},
       Descriptor{3, 1.515060824261142e-03, {+3.440503418178477e-01, +6.178873991598769e-01}},
       Descriptor{3, 5.911265567277567e-03, {+1.962412964100753e-01, +1.926549836686368e-01}},
       Descriptor{3, 8.392379781416604e-04, {+9.194790158373803e-01, +1.617257265834036e-02}},
       Descriptor{3, 4.022404010252260e-03, {+2.620925839988805e-01, +5.954881060323367e-01}},
       Descriptor{3, 2.231695755198927e-03, {+1.744887942814471e-01, +7.673520458998934e-01}},
       Descriptor{3, 6.156882008161937e-03, {+1.325632428580384e-01, +6.445714883912202e-01}},
       Descriptor{3, 8.642320589201948e-03, {+3.077710707446309e-01, +2.682338927247724e-01}},
       Descriptor{3, 7.452703387505539e-03, {+4.543788145456805e-01, +1.924966376990235e-01}},
       Descriptor{3, 5.028803835070171e-03, {+1.820389321231959e-01, +3.545120556149344e-01}},
       Descriptor{3, 8.796385122255454e-03, {+1.880443006828807e-01, +1.378439700992989e-01}},
       Descriptor{3, 1.747812460887048e-03, {+8.528586603669460e-01, +8.819642588395045e-02}},
       Descriptor{3, 9.189893838229260e-03, {+1.939535843143651e-01, +4.412634170405248e-01}},
       Descriptor{3, 6.208073919367888e-03, {+6.362056944924487e-01, +1.812670653663255e-01}},
       Descriptor{3, 5.683980552459499e-03, {+7.160962950323663e-01, +6.273138420630826e-02}},
       Descriptor{3, 1.836585716232118e-03, {+7.297058862326863e-01, +2.223880047949926e-01}},
       Descriptor{4, 2.218692647777552e-03, {+8.582850877054248e-01, +5.768403835499213e-01, +1.231132989726892e-01}},
       Descriptor{4, 2.609887405864026e-03, {+7.633676619034022e-01, +2.116780932527140e-01, +2.653658688975024e-02}},
       Descriptor{4, 6.168756108593601e-03, {+2.161303509635070e-01, +5.839223522583539e-01, +3.294970252944507e-01}},
       Descriptor{4, 1.229957612267995e-03, {+9.561584299674409e-01, +6.424182794168720e-01, +2.375108271645151e-02}},
       Descriptor{4, 1.954938784233806e-03, {+8.759221925259401e-01, +3.779756583128506e-01, +1.156156746353602e-02}},
       Descriptor{4, 3.467639337694073e-03, {+2.445394062188940e-01, +4.671941605403201e-01, +5.047812448827769e-01}},
       Descriptor{4, 6.770266043344008e-03, {+3.706058970009432e-01, +5.485269061480423e-01, +9.182493970820699e-02}},
       Descriptor{4, 1.678294163556885e-03, {+6.156798284057371e-01, +7.140171587890908e-03, +1.061641469954417e-02}},
       Descriptor{4, 7.130797590166880e-03, {+7.101327396551219e-01, +2.791894198719184e-01, +1.869924830554508e-01}},
       Descriptor{4, 4.675472898850534e-03, {+8.258711059677443e-01, +3.568302621331758e-01, +7.551964039133613e-02}},
       Descriptor{4, 2.534207524207739e-03, {+4.755766339064903e-01, +6.928649578239733e-01, +2.863100113252361e-01}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 16: {
    constexpr size_t nb_points = 285;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 2.767212020768290e-04, {+9.586035463620837e-01}},
       Descriptor{2, 7.349442577079450e-03, {+5.187038802297479e-01, +1.406555941365744e-01}},
       Descriptor{2, 1.487404370728813e-03, {+9.153129245676680e-01, +4.316491102619298e-02}},
       Descriptor{2, 3.876936384211876e-03, {+1.655472656731889e-02, +3.952283264601336e-01}},
       Descriptor{2, 1.030614824039442e-02, {+3.691213557507810e-01, +2.989012556159393e-01}},
       Descriptor{2, 2.726731988438958e-03, {+2.060330211667108e-02, +6.870461262955434e-02}},
       Descriptor{2, 2.446292134085587e-03, {+8.719419371010426e-01, +1.119349596786030e-01}},
       Descriptor{2, 5.549243997617444e-03, {+3.153842078793646e-01, +5.666543504043285e-01}},
       Descriptor{2, 5.022624174664197e-03, {+6.139955556315307e-01, +1.595773278635465e-01}},
       Descriptor{2, 6.711117008479383e-04, {+9.765007634893953e-02, +8.922021492300448e-01}},
       Descriptor{2, 3.168735186737549e-03, {+7.317646448078422e-01, +2.475499337151233e-01}},
       Descriptor{2, 2.307072686856537e-03, {+8.405422094352291e-01, +1.014565417563986e-02}},
       Descriptor{2, 2.233554340355750e-03, {+5.557460950562904e-01, +4.288551242077440e-01}},
       Descriptor{2, 2.286856931321262e-03, {+2.643802171735182e-01, +7.164979649536070e-01}},
       Descriptor{3, 5.838886999649284e-03, {+1.375503146864574e-01, +1.787752340257923e-01}},
       Descriptor{3, 4.120387561185287e-03, {+5.559992997537619e-01, +2.929550108496573e-01}},
       Descriptor{3, 1.730379481687403e-03, {+6.803760252907728e-01, +2.744752357294187e-01}},
       Descriptor{3, 4.106809066021676e-03, {+6.898707466220132e-01, +1.571412723940531e-01}},
       Descriptor{3, 6.354389480672592e-03, {+4.665601661638106e-01, +2.503592476644397e-01}},
       Descriptor{3, 4.326210184433355e-04, {+9.365742707871645e-01, +2.983913639912041e-02}},
       Descriptor{3, 3.452632929818020e-03, {+1.066846406851088e-01, +7.018965704311338e-01}},
       Descriptor{3, 4.240714557365053e-03, {+5.636286631903044e-01, +5.534643995880605e-02}},
       Descriptor{3, 3.405266316789942e-03, {+8.721470315267314e-02, +5.880318078594904e-01}},
       Descriptor{3, 7.009429293023371e-03, {+2.882648291123767e-01, +3.576004004129416e-01}},
       Descriptor{3, 3.711845099382991e-03, {+2.350324970540493e-01, +6.380849983494681e-01}},
       Descriptor{3, 6.557699894218194e-03, {+2.450858218595137e-01, +1.118004676798527e-01}},
       Descriptor{3, 1.699962978331957e-03, {+5.003064978621564e-01, +4.579960420164317e-01}},
       Descriptor{3, 1.347400048357294e-03, {+5.973130190403576e-02, +8.160093965927975e-01}},
       Descriptor{3, 3.633969504612614e-03, {+3.628596393971073e-01, +5.597294995869231e-02}},
       Descriptor{3, 1.216194629008639e-03, {+8.316301192301562e-01, +1.257310660131593e-01}},
       Descriptor{3, 5.495797083167567e-03, {+5.981515120789380e-01, +1.068019913435455e-01}},
       Descriptor{3, 9.733027356279389e-04, {+3.245246862338501e-01, +6.465862007088805e-01}},
       Descriptor{3, 7.253930032328553e-03, {+1.840043782869460e-01, +2.540941713633634e-01}},
       Descriptor{3, 2.616000670590770e-03, {+8.068558507667803e-01, +5.465918013108818e-02}},
       Descriptor{3, 5.483336923370839e-03, {+3.903817283942642e-01, +4.534593946231034e-01}},
       Descriptor{3, 1.501870645161511e-03, {+1.620565672046460e-01, +7.891766078049036e-01}},
       Descriptor{4, 2.242112763518706e-03, {+7.955779101854147e-01, +5.265131942175433e-01, +1.868978854107283e-01}},
       Descriptor{4, 2.543570121402526e-03, {+8.200116575745190e-01, +5.876389794788515e-01, +1.494308906905272e-02}},
       Descriptor{4, 5.361444814068662e-03, {+4.810677594905461e-01, +1.491370161533038e-01, +3.977024606563388e-01}},
       Descriptor{4, 1.505662384876206e-03, {+9.169801723597639e-01, +6.303394356928483e-01, +6.620609897561705e-02}},
       Descriptor{4, 1.126519980001580e-03, {+9.578199602162939e-01, +3.327783186859149e-01, +1.432824105432797e-02}},
       Descriptor{4, 2.836972153671841e-03, {+2.122679281909628e-01, +4.214367174540187e-01, +5.509963820368152e-01}},
       Descriptor{4, 4.234433041605569e-03, {+2.485957380035636e-01, +6.262321767993255e-01, +2.800030049831467e-01}},
       Descriptor{4, 3.184147857413263e-03, {+5.534790138573062e-01, +1.881580592010290e-01, +2.436330465584662e-01}},
       Descriptor{4, 4.029779987731148e-03, {+8.242282785245648e-01, +3.637620573617559e-01, +7.506931441576017e-02}},
       Descriptor{4, 3.669736816875404e-03, {+1.460654272875972e-01, +4.426831022376114e-01, +5.795059845476017e-02}},
       Descriptor{4, 2.488887287575317e-03, {+3.881072073502287e-01, +6.235512572087863e-01, +3.563115863132650e-01}},
       Descriptor{4, 4.888892260572901e-04, {+9.567448428329013e-01, +7.984451750366893e-01, +5.975255115121650e-03}},
       Descriptor{4, 3.252645155099476e-03, {+6.098484948586518e-01, +3.120515076281816e-01, +1.006650964023973e-02}},
       Descriptor{4, 4.955059091457157e-03, {+2.484742721540007e-01, +1.340653178754635e-01, +4.761968373134181e-01}},
       Descriptor{4, 4.570242180251667e-03, {+6.991894353535666e-01, +1.776787645087477e-01, +5.836635447459246e-02}},
       Descriptor{4, 4.810617642413540e-03, {+3.601541528165914e-01, +4.301177602083764e-01, +1.649556291845448e-01}},
       Descriptor{4, 5.387917947217286e-03, {+7.366874205051257e-01, +3.029873517367453e-01, +1.596398472135355e-01}},
       Descriptor{4, 2.469280567426453e-03, {+3.142480630211401e-01, +2.522065148266495e-02, +1.331460692424767e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 17: {
    constexpr size_t nb_points = 319;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.221682107890185e-04, {+9.722032338703692e-01}},
       Descriptor{1, 5.639436295815340e-03, {+6.387035292668141e-01}},
       Descriptor{1, 1.871168380432278e-03, {+6.802004320805534e-01}},
       Descriptor{2, 8.790055406139772e-04, {+9.524140858124460e-01, +4.194980824924494e-02}},
       Descriptor{2, 6.397368832390104e-04, {+8.275692125030436e-02, +9.014118320415641e-01}},
       Descriptor{2, 1.919424779584283e-03, {+2.657854504312773e-01, +6.843252932276896e-01}},
       Descriptor{2, 1.088568001837918e-03, {+8.115594002847951e-01, +8.245324867689100e-08}},
       Descriptor{2, 5.791783484828203e-03, {+3.523279995401680e-01, +5.541739087364920e-01}},
       Descriptor{2, 3.335679832473916e-03, {+6.030377121897242e-01, +3.625880598876809e-01}},
       Descriptor{2, 4.839934014148432e-03, {+7.840260420215905e-01, +1.387570850346267e-01}},
       Descriptor{2, 1.171223275540323e-02, {+4.934647219864639e-01, +2.426016748645620e-01}},
       Descriptor{2, 7.818032837412321e-03, {+3.503875164890664e-01, +4.013077580688794e-01}},
       Descriptor{2, 3.163898916366554e-03, {+3.364043121074268e-03, +3.815297326937233e-01}},
       Descriptor{2, 8.136278822739700e-03, {+4.307451267906745e-01, +1.190145691544862e-01}},
       Descriptor{2, 4.719826157526003e-03, {+5.925480128294769e-01, +4.928012549325394e-02}},
       Descriptor{2, 9.623713493802133e-04, {+2.255068603658533e-01, +7.620639151935824e-01}},
       Descriptor{3, 1.623787740135037e-03, {+6.397586374033618e-01, +3.115346949712161e-01}},
       Descriptor{3, 8.713459275997466e-03, {+1.646347487963126e-01, +2.565099172621796e-01}},
       Descriptor{3, 4.120086184663263e-03, {+4.930356331962961e-01, +7.449733603189460e-02}},
       Descriptor{3, 5.220647228935927e-04, {+7.479128039630975e-01, +1.307885480845669e-03}},
       Descriptor{3, 3.897164563433518e-03, {+2.625131786428597e-01, +5.285858021846196e-01}},
       Descriptor{3, 2.229921875344444e-03, {+1.748421558986807e-01, +9.552210724735053e-03}},
       Descriptor{3, 9.418067622583964e-03, {+2.760246878567057e-01, +3.414298990596975e-01}},
       Descriptor{3, 6.615832230310311e-04, {+4.351255240898762e-01, +2.702292596780164e-04}},
       Descriptor{3, 1.073942218989596e-04, {+9.530769415773807e-01, +4.692242455320396e-02}},
       Descriptor{3, 3.031478525882112e-03, {+4.842559790519491e-01, +3.444971328350538e-01}},
       Descriptor{3, 8.193264106692307e-04, {+2.938106184332757e-01, +6.681192301420841e-01}},
       Descriptor{3, 4.651214490706081e-03, {+1.605720050369202e-01, +6.739128715712450e-01}},
       Descriptor{3, 1.676067168115160e-03, {+8.101452921420736e-01, +3.437584591482228e-02}},
       Descriptor{3, 8.257689574656562e-03, {+1.800932575491559e-01, +1.554287257829705e-01}},
       Descriptor{3, 3.975458595443641e-03, {+5.013679940745993e-01, +2.697182873385593e-01}},
       Descriptor{3, 5.172252054798609e-04, {+9.228440785543659e-01, +1.181428825288093e-02}},
       Descriptor{3, 6.241363480381797e-03, {+3.207724957060760e-01, +6.366317285532444e-02}},
       Descriptor{3, 2.499541224162367e-03, {+2.145401504974506e-02, +6.762724457801919e-02}},
       Descriptor{3, 3.441345824053468e-03, {+6.529307874193767e-01, +2.014966460540852e-01}},
       Descriptor{3, 3.730857641897916e-03, {+3.810920673681604e-01, +4.858674466671989e-01}},
       Descriptor{3, 9.853795017806486e-03, {+4.141312471403079e-01, +1.907105246425098e-01}},
       Descriptor{3, 7.367579520284493e-03, {+1.431188794593187e-01, +5.103737585320696e-01}},
       Descriptor{3, 1.780487305606260e-03, {+6.816745760360433e-02, +8.035124551324651e-01}},
       Descriptor{3, 2.423131685196213e-03, {+2.297724242201254e-01, +4.121355156423638e-02}},
       Descriptor{3, 1.057340089483043e-03, {+7.821120027007296e-01, +1.776001617612250e-01}},
       Descriptor{3, 1.450363434917984e-03, {+4.764642149060935e-01, +4.843901243932818e-01}},
       Descriptor{3, 2.710579935745260e-03, {+6.655005676200287e-01, +4.796979480806327e-02}},
       Descriptor{3, 1.120145668844513e-03, {+1.575537160038546e-01, +8.010812345834409e-01}},
       Descriptor{4, 1.209171043596973e-03, {+7.476025549089913e-01, +1.335733555833585e-01, +2.410142270350212e-01}},
       Descriptor{4, 1.624713950736176e-03, {+6.402449723756475e-01, +3.697899872598127e-01, +1.069891762813818e-02}},
       Descriptor{4, 6.208589669146640e-03, {+2.772500272074544e-01, +6.588035927979838e-01, +2.398863138338967e-01}},
       Descriptor{4, 1.357245778023027e-03, {+8.843596956349281e-01, +7.690144551026831e-01, +8.514410320936129e-02}},
       Descriptor{4, 7.768823018224295e-04, {+9.543208345441746e-01, +2.756529256636570e-01, +8.471105348916165e-03}},
       Descriptor{4, 1.905388500388057e-03, {+1.961068637617923e-01, +4.881593428940726e-01, +4.956224506553615e-01}},
       Descriptor{4, 1.836355855674232e-03, {+5.253928688656228e-01, +9.036540229831336e-01, +4.642983560786509e-02}},
       Descriptor{4, 1.733122655911971e-03, {+7.871050645706265e-01, +5.305666908542513e-01, +1.965615185289643e-01}},
       Descriptor{4, 8.755868144016887e-04, {+8.730512492238405e-01, +5.824961459878540e-01, +6.875723092085683e-03}},
       Descriptor{4, 5.299199352810601e-03, {+4.953985501065977e-01, +2.191018460614218e-01, +3.867051301301994e-01}},
       Descriptor{4, 1.909231403394242e-03, {+8.695725428246946e-01, +3.055722326033879e-01, +1.089610679849025e-01}},
       Descriptor{4, 3.332703296489462e-03, {+8.255682153916121e-01, +1.820166521926553e-01, +4.416884971526350e-02}},
       Descriptor{4, 1.372582739425581e-03, {+2.263367583189702e-01, +3.553849916667940e-01, +6.248925484789940e-01}},
       Descriptor{4, 4.405628951728082e-03, {+5.513678656050841e-01, +7.486875952236592e-01, +1.172732959999338e-01}},
       Descriptor{4, 6.902657704569894e-03, {+6.416804127385760e-01, +2.589654916989230e-01, +1.195843625945489e-01}},
       Descriptor{4, 2.027344695276262e-03, {+4.862393601034299e-01, +1.128685901856588e-01, +1.562091696416078e-02}},
       Descriptor{4, 2.046435953340190e-03, {+4.272869183745079e-01, +6.283616414580556e-01, +3.513262055700426e-01}},
       Descriptor{4, 3.453264166213793e-04, {+9.787077639713484e-01, +7.341055271589447e-01, +1.323405399603627e-02}},
       Descriptor{4, 2.425089503329792e-03, {+7.058505258664157e-01, +4.135891617947305e-01, +2.986233869137579e-02}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 18: {
    constexpr size_t nb_points = 357;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.340052013254401e-04, {+9.689131877314755e-01}},
       Descriptor{2, 6.532829236534353e-03, {+7.386927583195001e-01, +1.121808531538677e-01}},
       Descriptor{2, 2.462677829860450e-03, {+7.680665730295287e-01, +2.095203731422156e-01}},
       Descriptor{2, 2.362663198775904e-03, {+3.167394480989801e-01, +4.504333920940983e-01}},
       Descriptor{2, 2.412846260195942e-03, {+6.187757221807219e-01, +3.671380924150245e-01}},
       Descriptor{2, 5.639399189884141e-03, {+3.723529440295544e-01, +5.149694555513888e-01}},
       Descriptor{2, 5.308794645373922e-03, {+5.038719357078286e-01, +2.465874064866041e-01}},
       Descriptor{2, 3.303209677609701e-03, {+7.220783650726965e-01, +3.347288009925681e-02}},
       Descriptor{2, 7.157096872930287e-03, {+3.940311335958268e-01, +1.760491594601057e-01}},
       Descriptor{2, 3.560209000799229e-03, {+1.181947980612448e-01, +1.262767090030246e-01}},
       Descriptor{2, 2.869089106153775e-03, {+5.117986204426422e-01, +1.146186842907397e-02}},
       Descriptor{2, 1.192387955426962e-03, {+9.344056049843052e-02, +8.286557181156362e-01}},
       Descriptor{2, 2.357089773027461e-04, {+9.778630528838397e-01, +4.114776635706134e-03}},
       Descriptor{2, 3.324152026523092e-03, {+2.689311802882549e-01, +6.643176008593026e-01}},
       Descriptor{2, 6.442449838790662e-04, {+1.858887671719376e-01, +8.071453164684427e-01}},
       Descriptor{3, 9.304082839731633e-04, {+8.803206212322914e-01, +6.759778909905437e-02}},
       Descriptor{3, 6.153126385517773e-04, {+6.814521213676156e-01, +2.940343113268007e-01}},
       Descriptor{3, 2.890748228612777e-04, {+4.969903643427416e-01, +4.966550809316037e-01}},
       Descriptor{3, 9.579280577883828e-03, {+3.063401808222882e-01, +2.791132054901511e-01}},
       Descriptor{3, 2.562433782912423e-03, {+5.853996645538505e-01, +7.876704807666435e-02}},
       Descriptor{3, 1.970097108482327e-03, {+7.769439547694348e-01, +4.825635484814583e-02}},
       Descriptor{3, 2.248538016231859e-03, {+1.869536840196541e-01, +8.812474861667117e-03}},
       Descriptor{3, 1.783785036670776e-03, {+6.440284406129158e-01, +2.005872107121198e-02}},
       Descriptor{3, 4.615488477875886e-04, {+1.993968440939729e-01, +7.563909496805638e-01}},
       Descriptor{3, 3.835673525848779e-03, {+4.915166272295902e-01, +1.515550593825411e-01}},
       Descriptor{3, 4.243233892689520e-03, {+2.197188373659789e-01, +6.084275881937196e-01}},
       Descriptor{3, 2.977098509258866e-03, {+3.818186027529240e-01, +5.256954545937250e-01}},
       Descriptor{3, 7.500976584211740e-03, {+1.323357443649693e-01, +4.028122600425206e-01}},
       Descriptor{3, 2.080642319051259e-03, {+2.643490668003158e-01, +4.148931197347632e-02}},
       Descriptor{3, 6.762965500915199e-03, {+2.625901672258577e-01, +1.466170076483845e-01}},
       Descriptor{3, 3.005874847678541e-03, {+6.368304996885981e-01, +2.566159742055003e-01}},
       Descriptor{3, 5.694731752763933e-03, {+5.033323003573335e-01, +2.555236724339244e-01}},
       Descriptor{3, 3.293452280749932e-03, {+4.305306648769874e-01, +3.698185671948173e-01}},
       Descriptor{3, 3.193430746935616e-03, {+4.412117799623165e-01, +1.554708555805914e-02}},
       Descriptor{3, 3.007859639579343e-03, {+1.233852653491083e-01, +4.951092940950103e-02}},
       Descriptor{3, 6.210180814260240e-03, {+2.819646674678744e-01, +4.572992412719536e-01}},
       Descriptor{3, 6.903801293936010e-03, {+1.230206594250918e-01, +2.586539281154879e-01}},
       Descriptor{3, 1.006183863763346e-04, {+9.699534085616927e-01, +1.471561063742728e-02}},
       Descriptor{3, 1.356307828233825e-04, {+1.536759742301584e-01, +8.430293850954313e-01}},
       Descriptor{3, 7.789376721955394e-04, {+3.232491600294373e-01, +6.422131631619533e-01}},
       Descriptor{3, 2.674414703364759e-03, {+7.979061002341173e-02, +7.081546894975584e-01}},
       Descriptor{3, 4.775459413065313e-04, {+5.572751587573097e-02, +9.075661676644768e-01}},
       Descriptor{3, 7.406610401180713e-04, {+8.040718007941732e-01, +1.580109891260832e-01}},
       Descriptor{3, 5.291370439452658e-03, {+1.063079492932114e-01, +5.586401442375718e-01}},
       Descriptor{3, 1.500664611055249e-03, {+1.496811655200803e-01, +7.655341121253637e-01}},
       Descriptor{3, 2.626273999362915e-03, {+7.217980235897704e-01, +1.203862817704082e-01}},
       Descriptor{4, 1.065436899176937e-03, {+8.179717082248776e-01, +6.432021548505883e-01, +1.654004245581700e-01}},
       Descriptor{4, 6.080722561497974e-03, {+4.547777230252971e-01, +1.663641899799587e-01, +7.191647566286845e-02}},
       Descriptor{4, 5.310023010585634e-03, {+2.207750902147102e-01, +6.491941928630117e-01, +2.436936622868156e-01}},
       Descriptor{4, 2.020033424832363e-03, {+8.584914437706522e-01, +2.644232091625772e-01, +3.997911405124788e-02}},
       Descriptor{4, 2.164703930837454e-03, {+1.908724705162439e-01, +4.565598584631854e-01, +5.240352269645050e-01}},
       Descriptor{4, 4.353306989152711e-03, {+3.778781661386005e-01, +6.640629325475310e-01, +6.332251756914624e-02}},
       Descriptor{4, 6.863403477821615e-04, {+8.939555664639546e-01, +1.728332917834724e-01, +6.271673653011652e-03}},
       Descriptor{4, 4.019067831153739e-03, {+4.099141505393519e-01, +5.003413197772252e-02, +3.461255047044217e-01}},
       Descriptor{4, 7.912728473970523e-04, {+9.319810865442065e-01, +1.066217398606188e-01, +4.934944134498662e-02}},
       Descriptor{4, 2.446710528967660e-03, {+2.801822060720651e-01, +8.499319598680132e-01, +1.205025919278488e-01}},
       Descriptor{4, 1.950381902056960e-03, {+5.772816541361394e-01, +4.620142974473129e-01, +3.942812057164277e-01}},
       Descriptor{4, 1.105915938820858e-03, {+8.445158080852770e-01, +6.175679175506062e-01, +8.380194537906345e-03}},
       Descriptor{4, 5.030114823747663e-03, {+5.294157289381186e-01, +2.334853358468202e-01, +3.799152175151670e-01}},
       Descriptor{4, 6.596708157003132e-04, {+9.282752180189467e-01, +6.794178627981728e-01, +6.658426856019066e-02}},
       Descriptor{4, 6.132163978139013e-04, {+9.683651638409505e-01, +4.614425143846638e-01, +1.360996700708025e-02}},
       Descriptor{4, 8.965714244660760e-04, {+1.674457184003009e-01, +3.143041479415947e-01, +6.737791163005959e-01}},
       Descriptor{4, 3.639835604771190e-03, {+4.923271845449609e-01, +7.398884292540342e-01, +1.512239686451382e-01}},
       Descriptor{4, 2.111289190280422e-03, {+8.565997422827011e-01, +5.406106561720058e-01, +5.960704457999711e-02}},
       Descriptor{4, 5.700622245430812e-03, {+2.670227736126091e-01, +5.753139352915381e-01, +1.555725209349524e-01}},
       Descriptor{4, 1.955327766010427e-03, {+4.150645766534223e-01, +7.173101873515257e-01, +2.646731283533919e-01}},
       Descriptor{4, 6.238495350617601e-04, {+9.446623957916176e-01, +8.161908018659551e-01, +1.193065095103094e-02}},
       Descriptor{4, 1.517921654020750e-03, {+7.145052262408169e-01, +3.117978322705600e-01, +7.294130256492850e-03}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 19: {
    constexpr size_t nb_points = 418;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 1.287002239747240e-04, {+9.673913352370350e-01}},
       Descriptor{1, 5.390761254300309e-03, {+5.436653246463873e-01}},
       Descriptor{2, 6.666871057805598e-03, {+3.736572163452879e-01, +4.188756062275946e-01}},
       Descriptor{2, 6.198011213798800e-03, {+4.375282703042845e-01, +2.977489444568584e-01}},
       Descriptor{2, 1.196903942172590e-03, {+5.578275629324269e-01, +4.363255890435926e-01}},
       Descriptor{2, 9.000351976243133e-03, {+2.727587922914985e-01, +1.312347586523022e-01}},
       Descriptor{2, 4.164637105384020e-03, {+2.887241175382593e-01, +5.688216954681572e-01}},
       Descriptor{2, 1.824497208727276e-03, {+1.127746562325076e-01, +6.623057363188606e-01}},
       Descriptor{2, 4.523718683330509e-03, {+6.547023041522421e-01, +2.196378912459858e-01}},
       Descriptor{2, 8.161116064814124e-03, {+2.229510128266209e-01, +2.273269633283456e-01}},
       Descriptor{2, 1.648326828637079e-03, {+3.517668912115330e-01, +5.417515126253033e-03}},
       Descriptor{2, 5.571492309500324e-04, {+2.069773541366567e-01, +7.929490613648335e-01}},
       Descriptor{2, 1.118215252194827e-03, {+9.311118902772667e-01, +4.180662980331772e-02}},
       Descriptor{2, 1.885810158761153e-03, {+8.430335844595135e-01, +1.361551806457953e-01}},
       Descriptor{2, 2.828101226263676e-04, {+7.303948013428374e-02, +9.187547560280617e-01}},
       Descriptor{2, 1.815863546152488e-03, {+6.780735133862854e-01, +3.019190562178010e-01}},
       Descriptor{2, 1.247082448220326e-03, {+8.819848262866421e-01, +9.387499849089011e-03}},
       Descriptor{2, 1.074162553027278e-03, {+3.858781168610365e-01, +6.065859845746238e-01}},
       Descriptor{2, 2.579770777378314e-03, {+1.370888473751836e-01, +7.535348220210553e-01}},
       Descriptor{3, 9.170154497260322e-04, {+5.255852110826873e-01, +4.408685369060898e-01}},
       Descriptor{3, 3.068778721173566e-03, {+4.530164257058980e-01, +4.188977508155631e-01}},
       Descriptor{3, 1.871058171917288e-03, {+6.712511632017596e-01, +1.251810432139165e-02}},
       Descriptor{3, 1.536909390806274e-04, {+3.695141864916882e-01, +6.246014838381103e-01}},
       Descriptor{3, 3.845290732110408e-03, {+2.883822343558625e-01, +5.439973273857342e-02}},
       Descriptor{3, 2.243116164244068e-04, {+8.285723838946580e-01, +1.619186219316668e-01}},
       Descriptor{3, 8.402087626133961e-04, {+6.724607008544720e-01, +2.930033138342824e-01}},
       Descriptor{3, 3.041957674540431e-03, {+4.873668933896256e-01, +2.634265889387019e-01}},
       Descriptor{3, 2.294841766987932e-04, {+9.503664431507984e-01, +1.127614473867753e-02}},
       Descriptor{3, 4.182420591869638e-03, {+9.132353058612172e-02, +3.580798461499961e-01}},
       Descriptor{3, 6.327091081505453e-03, {+1.552069555141659e-01, +4.714188671670859e-01}},
       Descriptor{3, 7.592834005951271e-04, {+5.052806321356114e-02, +8.544640711209540e-01}},
       Descriptor{3, 4.596836662276725e-03, {+3.978139280416619e-01, +1.293031169145601e-01}},
       Descriptor{3, 2.766364274235799e-03, {+3.760791314064199e-01, +7.899233073306565e-02}},
       Descriptor{3, 2.144850801510551e-03, {+1.643638267982826e-01, +1.672886065733771e-02}},
       Descriptor{3, 5.283144486182838e-03, {+3.468892555686117e-01, +3.885050437237611e-01}},
       Descriptor{3, 4.881052322614178e-04, {+1.191332579371944e-01, +8.475473975749326e-01}},
       Descriptor{3, 2.120695458348499e-03, {+7.454583814188197e-01, +1.626360280018163e-01}},
       Descriptor{3, 1.615060199482018e-03, {+7.610495209730348e-01, +7.862268330374027e-02}},
       Descriptor{3, 6.942257092948444e-03, {+2.420812058877188e-01, +3.202806437619411e-01}},
       Descriptor{3, 1.002683020081214e-03, {+8.445798381566891e-01, +1.382133423647079e-02}},
       Descriptor{3, 9.207753167491697e-04, {+1.645646578166248e-01, +7.687752250435795e-01}},
       Descriptor{3, 3.556169477445426e-03, {+2.747842347238860e-01, +5.301402408730526e-01}},
       Descriptor{3, 2.186178713296525e-03, {+5.849919450560005e-01, +3.008923824441204e-01}},
       Descriptor{3, 2.691651662032098e-03, {+1.489738124603387e-01, +6.167319270078593e-01}},
       Descriptor{3, 3.016740104757321e-03, {+4.389290397617306e-01, +1.376377254881576e-02}},
       Descriptor{3, 7.466645241550527e-04, {+8.798197120003726e-01, +7.210305012229555e-02}},
       Descriptor{3, 2.997482336902333e-03, {+1.168533193079235e-01, +6.238118616203953e-02}},
       Descriptor{3, 1.447847026063246e-03, {+3.428716485856084e-01, +5.855795349165888e-01}},
       Descriptor{4, 1.072809629353361e-03, {+8.643327984186001e-01, +6.422236743536752e-01, +1.179192064779849e-01}},
       Descriptor{4, 4.839037348378018e-03, {+5.784047959665971e-01, +1.570037147921136e-01, +1.114252341292851e-01}},
       Descriptor{4, 2.747099059820038e-03, {+3.108799114935293e-01, +5.391600980992765e-01, +2.750267281218793e-01}},
       Descriptor{4, 4.335016889029408e-03, {+2.047256512125053e-01, +5.650506986268603e-01, +3.472846972655921e-01}},
       Descriptor{4, 2.718100159849547e-03, {+1.547714831454746e-01, +8.008730085913268e-01, +7.549551730236130e-02}},
       Descriptor{4, 9.168547398610547e-04, {+9.112222485018308e-01, +3.619692710244549e-01, +3.464296962031117e-02}},
       Descriptor{4, 6.808951019688473e-03, {+2.376393444235864e-01, +4.605447477525737e-01, +2.044152977923567e-01}},
       Descriptor{4, 4.013856896610910e-04, {+9.302667501209977e-01, +3.215980764268125e-01, +6.972695561067982e-02}},
       Descriptor{4, 2.155002895452285e-03, {+2.733067099141755e-01, +7.541757880282322e-01, +2.199898046581558e-01}},
       Descriptor{4, 1.611542573023053e-03, {+6.197790831123378e-01, +4.009499952384629e-01, +3.646225494795918e-01}},
       Descriptor{4, 2.743455266744436e-04, {+9.673179731072893e-01, +7.175030678238746e-01, +4.586334961223785e-03}},
       Descriptor{4, 2.300082975062185e-03, {+2.810759624172646e-01, +1.446964075338812e-01, +6.664230048871306e-01}},
       Descriptor{4, 2.945835759051771e-04, {+9.572349478971794e-01, +8.054431343183734e-01, +3.975302308577006e-02}},
       Descriptor{4, 6.007312216772520e-04, {+9.341504389265958e-01, +5.642632183544610e-01, +3.073598825835916e-02}},
       Descriptor{4, 2.620175037577330e-04, {+2.212443839381749e-01, +2.763723337715620e-01, +7.207735178967332e-01}},
       Descriptor{4, 4.192832498709637e-03, {+5.144086015486090e-01, +6.711096936374128e-01, +1.403609469612069e-01}},
       Descriptor{4, 1.736294308688135e-03, {+8.466968176319485e-01, +6.476309524361428e-01, +5.264191965632582e-02}},
       Descriptor{4, 9.439101669434532e-04, {+5.837234015734821e-01, +7.587849158481068e-01, +2.270233248942167e-01}},
       Descriptor{4, 2.175513608503740e-03, {+2.312548427820351e-01, +7.579578663965916e-01, +2.881326163838643e-02}},
       Descriptor{4, 1.283394690183355e-03, {+6.535198164687628e-01, +2.118649081243290e-01, +5.321389368184961e-03}},
       Descriptor{4, 2.726150499605700e-03, {+6.953450152489690e-01, +1.860513163719893e-01, +1.591709313067350e-01}},
       Descriptor{4, 3.185339603850620e-03, {+8.127234831798691e-02, +5.241033074238067e-01, +4.286780869626788e-02}},
       Descriptor{4, 2.663092866644570e-03, {+6.731624418813449e-01, +4.707459336103425e-01, +2.281921560389168e-01}},
       Descriptor{4, 2.326668465288244e-03, {+8.297715217071255e-01, +3.845634575264398e-01, +1.149758475625305e-01}},
       Descriptor{4, 9.734631151970315e-04, {+8.530842873329166e-01, +4.855129905435616e-01, +6.546603573639168e-03}},
       Descriptor{4, 3.483456851399480e-03, {+4.467073185069126e-01, +1.775709599102257e-01, +4.889658061311413e-01}},
       Descriptor{4, 4.061954216963042e-03, {+6.700967212030146e-01, +4.619994161131860e-01, +5.843691849737929e-02}},
       Descriptor{4, 3.324559356884226e-04, {+9.767336037107472e-01, +2.679001927626974e-01, +6.104985413274207e-03}},
       Descriptor{4, 9.232817417531915e-04, {+3.188790007212526e-01, +4.538176463136764e-01, +5.340866401557672e-01}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
  case 20: {
    constexpr size_t nb_points = 489;
    SmallArray<R3> point_list(nb_points);
    SmallArray<double> weight_list(nb_points);

    std::array descriptor_list =   //
      {Descriptor{1, 5.374309731479196e-03, {+6.041261328156059e-01}},
       Descriptor{2, 5.671734293778256e-03, {+4.816935434431252e-01, +1.283254121370188e-01}},
       Descriptor{2, 3.248860881724051e-03, {+6.985803886987003e-01, +2.363645279387677e-01}},
       Descriptor{2, 3.452094526682737e-03, {+5.346865795177097e-01, +3.694814396873626e-01}},
       Descriptor{2, 6.666033444769521e-04, {+4.222838087844267e-01, +5.741352229223836e-01}},
       Descriptor{2, 3.995792460381601e-03, {+3.068045188877912e-01, +5.709702769195857e-01}},
       Descriptor{2, 4.992778959530078e-03, {+2.287749849768982e-01, +4.453272633438413e-01}},
       Descriptor{2, 2.469622976420997e-03, {+5.538968741280718e-01, +1.850719926469714e-01}},
       Descriptor{2, 7.066155754130616e-03, {+3.848922804233133e-01, +2.828189872249547e-01}},
       Descriptor{2, 1.916980035371976e-03, {+2.571105687384414e-01, +7.453813550512145e-03}},
       Descriptor{2, 2.387541558193644e-04, {+9.495420833471157e-01, +2.730608888531540e-03}},
       Descriptor{2, 1.680239670130047e-04, {+9.635090648066924e-02, +8.947468307319109e-01}},
       Descriptor{2, 9.944327373233232e-04, {+9.104493126181845e-01, +5.489260378459437e-02}},
       Descriptor{2, 3.454941677138827e-03, {+2.683082435742811e-01, +1.789054761711333e-01}},
       Descriptor{2, 1.377780657771690e-03, {+8.274465668629972e-01, +1.443872792953364e-01}},
       Descriptor{2, 6.135678341459227e-05, {+3.789270083108932e-02, +9.600532902975129e-01}},
       Descriptor{2, 9.961596076112076e-04, {+5.735205240815813e-01, +4.226237395657491e-01}},
       Descriptor{2, 1.734688844684138e-03, {+7.423115722700304e-01, +9.373411784322374e-03}},
       Descriptor{2, 6.408905290179002e-04, {+2.846909641320057e-01, +7.051164480862681e-01}},
       Descriptor{2, 1.747701749929037e-03, {+1.128162795069089e-01, +8.097603763752680e-01}},
       Descriptor{3, 5.356183098283481e-04, {+4.535676239967043e-01, +5.210700225779646e-01}},
       Descriptor{3, 3.744128155669667e-03, {+3.336110385261160e-01, +4.762219126204001e-01}},
       Descriptor{3, 3.648498543938937e-03, {+3.773337527822211e-01, +4.553198949867700e-02}},
       Descriptor{3, 2.722104630492699e-04, {+3.061117491692780e-01, +6.823651677845297e-01}},
       Descriptor{3, 4.038058627934289e-03, {+2.904460076524542e-01, +3.676651830752369e-01}},
       Descriptor{3, 2.849455013152634e-03, {+5.914106884092367e-01, +3.140155892114704e-02}},
       Descriptor{3, 4.946318515118939e-04, {+7.660549649135295e-01, +2.085476317987768e-01}},
       Descriptor{3, 3.435750243515581e-03, {+4.496216926317984e-01, +3.209594399518608e-01}},
       Descriptor{3, 7.654793642697566e-04, {+6.125691075458759e-01, +3.559054105435636e-01}},
       Descriptor{3, 9.270297025443059e-05, {+9.651777563788078e-01, +1.963481046583392e-02}},
       Descriptor{3, 4.440447400598151e-03, {+1.850944308523854e-01, +5.811084832537615e-01}},
       Descriptor{3, 4.995284180274901e-03, {+1.305339914072845e-01, +2.203018355115161e-01}},
       Descriptor{3, 2.327605643441156e-03, {+8.713330755302490e-02, +7.074337461014996e-01}},
       Descriptor{3, 3.058461110576954e-03, {+1.162657955161678e-01, +4.975461839293467e-01}},
       Descriptor{3, 2.716703334283396e-04, {+3.353132384174681e-02, +9.019799487990967e-01}},
       Descriptor{3, 3.592040414552888e-03, {+6.300704464975707e-01, +1.583017481238254e-01}},
       Descriptor{3, 4.506232324626833e-03, {+1.008155351127816e-01, +3.470348929613336e-01}},
       Descriptor{3, 2.662059522271117e-04, {+9.712721271803038e-02, +8.772759077750505e-01}},
       Descriptor{3, 1.722422251895809e-03, {+6.503674461202499e-01, +2.509808878874181e-01}},
       Descriptor{3, 1.529902163082495e-03, {+7.881670555390969e-01, +1.228748706112481e-01}},
       Descriptor{3, 4.331723455652033e-03, {+4.632610253517699e-01, +2.156955721524682e-01}},
       Descriptor{3, 8.698495590594387e-04, {+8.742622562545352e-01, +4.316688220319717e-02}},
       Descriptor{3, 3.166557615941483e-03, {+9.693365579654889e-02, +1.072427543090557e-01}},
       Descriptor{3, 1.455167758465003e-03, {+5.265123436991546e-01, +3.296427782470148e-01}},
       Descriptor{3, 1.137616034931676e-03, {+2.105801835055998e-01, +7.160068077591689e-01}},
       Descriptor{3, 1.884904388466294e-03, {+4.834178601814259e-01, +4.197363107268692e-01}},
       Descriptor{3, 6.049803062524013e-03, {+2.723239805399290e-01, +1.077197094023458e-01}},
       Descriptor{3, 2.785053160390842e-04, {+8.874839070114657e-01, +9.137076099343852e-02}},
       Descriptor{3, 1.644290914957632e-03, {+3.400434017708192e-01, +5.743322455524006e-01}},
       Descriptor{3, 4.384669874639433e-03, {+2.465713452115888e-01, +2.965278956806727e-01}},
       Descriptor{3, 2.508217360325574e-03, {+7.203261185246220e-01, +6.861567128774998e-02}},
       Descriptor{4, 6.403192997163759e-04, {+8.693467247184354e-01, +6.605377674376229e-01, +1.211268750412950e-01}},
       Descriptor{4, 2.743609819133677e-03, {+5.165700364414527e-01, +1.268114432243165e-01, +6.232922095164157e-02}},
       Descriptor{4, 4.425132157331708e-03, {+2.141949283591463e-01, +5.351420520651734e-01, +2.887837560282615e-01}},
       Descriptor{4, 3.793533547619605e-03, {+3.860615525036007e-01, +1.430987462436435e-01, +4.198199632798237e-01}},
       Descriptor{4, 2.598731880891428e-03, {+3.148265223116505e-01, +5.788734905442336e-01, +3.671971633512633e-01}},
       Descriptor{4, 1.100562071404283e-03, {+1.333776220074420e-01, +8.685734808776638e-01, +3.539982573087990e-02}},
       Descriptor{4, 7.559451274546276e-04, {+8.933451426565783e-01, +2.970093961186454e-01, +6.894095497062928e-03}},
       Descriptor{4, 4.164432957175231e-03, {+6.420732128180285e-01, +2.185904390936289e-01, +1.803409782254840e-01}},
       Descriptor{4, 7.986472006319258e-04, {+9.015566925902675e-01, +2.850450105795807e-01, +8.701177937660884e-02}},
       Descriptor{4, 1.274772831709237e-03, {+1.795078173861985e-01, +6.796606097350029e-01, +3.018403418988506e-01}},
       Descriptor{4, 7.742072015194865e-04, {+5.287716454511575e-01, +3.752537914501156e-01, +4.637531647615307e-01}},
       Descriptor{4, 1.079867868775378e-03, {+8.568477683810934e-01, +7.061426122653948e-01, +1.259447070049874e-02}},
       Descriptor{4, 1.550935782403934e-03, {+2.374464971047839e-01, +9.039264484495425e-02, +6.846455484276207e-01}},
       Descriptor{4, 4.993786208816673e-04, {+9.406598770106565e-01, +7.437511960991476e-01, +4.669618994654997e-02}},
       Descriptor{4, 4.262366420646442e-04, {+9.654498557948427e-01, +5.581648667244832e-01, +9.979099797204329e-03}},
       Descriptor{4, 5.244528523523384e-04, {+1.185997715939458e-01, +1.973055345489109e-01, +7.917076093806301e-01}},
       Descriptor{4, 2.954838737142077e-03, {+4.383682874673118e-01, +6.801053445689546e-01, +2.323974680712023e-01}},
       Descriptor{4, 1.540648448261571e-03, {+8.501501583895467e-01, +5.776785987998531e-01, +6.212941698545271e-02}},
       Descriptor{4, 4.849861236258662e-03, {+4.097788353774976e-01, +5.992649167860331e-01, +1.025000747100843e-01}},
       Descriptor{4, 7.495372210984812e-04, {+6.254025057545211e-01, +7.793811396398747e-01, +1.963810536401145e-01}},
       Descriptor{4, 1.677329035907760e-03, {+1.658794771155705e-01, +6.513788165859978e-01, +3.807225396126873e-02}},
       Descriptor{4, 2.242040914713952e-03, {+5.062559144366789e-01, +2.470826877566723e-01, +9.699938299695787e-03}},
       Descriptor{4, 3.054238951987089e-03, {+7.429293632807329e-01, +1.682359345705867e-01, +8.712730173221997e-02}},
       Descriptor{4, 2.363709150448725e-03, {+1.867609649526593e-02, +2.536204599781817e-01, +4.169093747605618e-02}},
       Descriptor{4, 4.124773767916695e-03, {+2.608600527928172e-01, +4.094605558894180e-01, +1.898222236268910e-01}},
       Descriptor{4, 2.522814373870828e-03, {+7.938808312983506e-01, +4.691881899824492e-01, +1.276217397945529e-01}},
       Descriptor{4, 1.143364037200234e-03, {+7.930116554406030e-01, +2.986457776503003e-01, +1.955063271970715e-01}},
       Descriptor{4, 2.785004192577059e-03, {+4.492999606983316e-01, +1.696796572740249e-01, +4.935677375430703e-01}},
       Descriptor{4, 2.222625782566179e-03, {+7.687383145879386e-01, +3.967745055048160e-01, +3.472481178210238e-02}},
       Descriptor{4, 2.609946044201681e-04, {+9.785997597757148e-01, +1.909470577592247e-01, +1.518094278770999e-02}},
       Descriptor{4, 1.252174210689572e-03, {+2.135890927684384e-01, +3.604161529615568e-01, +6.154203152262100e-01}},
       Descriptor{4, 1.279670338642788e-03, {+7.962525773875013e-01, +1.562852697636821e-01, +1.257520475253165e-01}},
       Descriptor{4, 7.990495656729902e-04, {+9.193985656269752e-01, +4.148607451061945e-01, +4.457888016467764e-02}},
       Descriptor{4, 7.812219859741840e-04, {+5.150733025160915e-01, +6.896293007082364e-01, +3.017078195063375e-01}},
       Descriptor{4, 2.172588051809931e-04, {+9.636121389116856e-01, +8.595767942792535e-01, +5.771284380553639e-03}},
       Descriptor{4, 7.595577614438183e-04, {+7.214071149093607e-01, +4.920954370020814e-01, +2.395309192181843e-03}}};

    fill_quadrature_points(descriptor_list, point_list, weight_list);

    m_point_list  = point_list;
    m_weight_list = weight_list;
    break;
  }
    // LCOV_EXCL_START
  default: {
    throw UnexpectedError("Gauss quadrature formulae handle degrees up to " +
                          std::to_string(PyramidGaussQuadrature::max_degree) + " on pyramids");
  }
    // LCOV_EXCL_STOP
  }
}
